/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql.cqn;

import java.util.function.Supplier;

import com.google.common.annotations.Beta;
import com.sap.cds.Cds4jServiceLoader;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.reflect.CdsModel;

/**
 * API to introspect {@link CqnStructuredTypeRef CqnStructuredTypeRef(s)} and
 * {@link CqnStatement CqnStatement(s)}
 */
public final class CqnAnalyzer {

	private static final CqnAnalyzerSPI analyzer = Cds4jServiceLoader.load(CqnAnalyzerSPI.class);

	private final Supplier<CdsModel> modelSupplier;

	private CqnAnalyzer(Supplier<CdsModel> modelSupplier) {
		this.modelSupplier = modelSupplier;
	}

	/**
	 * Creates an instance of {@code CqnAnalyzer}.
	 * 
	 * @param model the {@link CdsModel}
	 * @return the {@code CqnAnalyzer} instance
	 */
	public static CqnAnalyzer create(CdsModel model) {
		return create(() -> model);
	}

	/**
	 * Creates an instance of {@code CqnAnalyzer}.
	 * 
	 * @param modelSupplier the {@link CdsModel} supplier
	 * @return the {@code CqnAnalyzer} instance
	 */
	public static CqnAnalyzer create(Supplier<CdsModel> modelSupplier) {
		return new CqnAnalyzer(modelSupplier);
	}

	/**
	 * Analyzes the {@link Segment Segment(s)} of the given
	 * {@link CqnStructuredTypeRef}.
	 * 
	 * @param ref the {@code CqnStructuredTypeRef} to be analyzed
	 * @return the {@link AnalysisResult} with information on the segments of the
	 *         {@code CqnStructuredTypeRef}
	 */
	public AnalysisResult analyze(CqnStructuredTypeRef ref) {
		return analyzer.analyze(modelSupplier.get(), ref);
	}

	/**
	 * Analyzes the {@code ref} and {@code where} condition of the given
	 * {@link CqnSelect}.
	 * 
	 * @param select the {@code CqnSelect} statement to be analyzed
	 * @return the {@link AnalysisResult} with information on the {@code ref}
	 *         {@link Segment Segment(s)} and {@code where} condition of the select
	 */
	public AnalysisResult analyze(CqnSelect select) {
		return analyzer.analyze(modelSupplier.get(), select);
	}

	/**
	 * Analyzes the {@code ref} and {@code where} condition of the given
	 * {@link CqnUpdate}. The update data is not considered.
	 * 
	 * @param update the {@code CqnUpdate} statement to be analyzed
	 * @return the {@link AnalysisResult} with information on the {@code ref}
	 *         {@link Segment Segment(s)} and {@code where} condition of the update
	 */
	public AnalysisResult analyze(CqnUpdate update) {
		return analyzer.analyze(modelSupplier.get(), update);
	}

	/**
	 * Analyzes the {@code ref} and {@code where} condition of the given
	 * {@link CqnDelete}.
	 * 
	 * @param delete the {@code CqnDelete} statement to be analyzed
	 * @return the {@link AnalysisResult} with information on the {@code ref}
	 *         {@link Segment Segment(s)} and {@code where} condition of the delete
	 */
	public AnalysisResult analyze(CqnDelete delete) {
		return analyzer.analyze(modelSupplier.get(), delete);
	}

	@Beta
	public Iterable<ResolvedRefItem> resolveRefItems(CqnSelect query) {
		return analyzer.resolveRefItems(modelSupplier.get(), query);
	}

	/**
	 * Service Provider Interface for the {@link CqnAnalyzer}.
	 */
	public interface CqnAnalyzerSPI {

		AnalysisResult analyze(CdsModel model, CqnStructuredTypeRef ref);

		AnalysisResult analyze(CdsModel model, CqnSelect select);

		AnalysisResult analyze(CdsModel model, CqnUpdate delete);

		AnalysisResult analyze(CdsModel model, CqnDelete delete);

		Iterable<ResolvedRefItem> resolveRefItems(CdsModel model, CqnSelect query);
	}

}
