/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql.cqn;

import java.time.temporal.Temporal;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.Expand;
import com.sap.cds.ql.Limit;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.Value;

/**
 * Allows to modify CQN statements copied with
 * {@link CQL#copy(CqnStatement statement, CqnModifier modifier) CQL.copy}.
 */
public interface CqnModifier {

	default Value<?> ref(ElementRef<?> ref) {
		return ref;
	}

	default CqnStructuredTypeRef ref(StructuredTypeRef ref) {
		return ref;
	}

	default Value<?> literal(Object value, String cdsType) {
		return CQL.literal(value).type(cdsType);
	}

	default Value<?> literal(String val, String cdsType) {
		return literal((Object) val, cdsType);
	}

	default Value<?> literal(Number number, String cdsType) {
		return literal((Object) number, cdsType);
	}

	default Value<?> literal(Boolean bool, String cdsType) {
		return literal((Object) bool, cdsType);
	}

	default Value<?> literal(Temporal temporal, String cdsType) {
		return literal((Object) temporal, cdsType);
	}

	default Value<?> parameter(String name, String cdsType) {
		return CQL.param(name).type(cdsType);
	}

	default Value<?> plain(String plain, String cdsType) {
		return CQL.plain(plain).type(cdsType);
	}

	default Value<?> function(String name, List<Value<?>> args, String cdsType) {
		return CQL.func(name, args).type(cdsType);
	}

	default Predicate booleanFunction(String name, List<Value<?>> args) {
		return CQL.booleanFunc(name, args);
	}

	default Predicate comparison(Value<?> lhs, CqnComparisonPredicate.Operator op, Value<?> rhs) {
		return CQL.comparison(lhs, op, rhs);
	}

	default Predicate in(Value<?> value, Collection<Value<?>> list) {
		return CQL.in(value, list);
	}

	default Predicate connective(CqnConnectivePredicate.Operator op, List<Predicate> predicates) {
		if (op == CqnConnectivePredicate.Operator.AND) {
			return CQL.and(predicates);
		}
		return CQL.or(predicates);
	}

	default Predicate negation(Predicate p) {
		return CQL.not(p);
	}

	default Predicate search(String term) {
		return CQL.search(term);
	}

	default Predicate exists(Select<?> subQuery) {
		return CQL.exists(subQuery);
	}

	default CqnSelectListItem selectListItem(Value<?> value, String alias) {
		return value.as(alias);
	}

	default CqnSelectListItem selectAll() {
		return CQL.star();
	}

	default CqnSelectListItem inline(StructuredTypeRef ref, List<CqnSelectListItem> items) {
		return CQL.to(ref.segments()).inline(items);
	}

	default CqnSelectListItem expand(StructuredTypeRef ref, List<CqnSelectListItem> items,
			List<CqnSortSpecification> orderBy, CqnLimit limit) {
		Expand<?> expand = CQL.to(ref.segments()).expand(items).orderBy(orderBy);
		if (limit != null) {
			expand.limit(limit.top(), limit.skip());
		}
		return expand;
	}

	default List<CqnSelectListItem> items(List<CqnSelectListItem> items) {
		return items;
	}

	default Set<String> excluding(Set<String> excluding) {
		return excluding;
	}

	default boolean distinct(boolean distinct) {
		return distinct;
	}

	default boolean inlineCount(boolean inlineCount) {
		return inlineCount;
	}

	default Predicate where(Predicate where) {
		return where;
	}

	default Predicate search(Predicate search) {
		return search;
	}

	default Set<String> searchableElements(Set<String> searchableElements) {
		return searchableElements;
	}

	default List<CqnSelectListItem> groupBy(List<CqnSelectListItem> groupBy) {
		return groupBy;
	}

	default Predicate having(Predicate having) {
		return having;
	}

	default List<CqnSortSpecification> orderBy(List<CqnSortSpecification> orderBy) {
		return orderBy;
	}

	default CqnLimit limit(Limit limit) {
		return limit;
	}

}
