/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.reflect;

import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Stream;

public interface CdsAnnotatable {

	/**
	 * Returns a sequential {@code Stream} over all {@link CdsAnnotation
	 * CdsAnnotation(s)} of this {@code CdsAnnotatable}.
	 *
	 * @return a sequential {@code Stream} over the {@code CdsAnnotation(s)}
	 */
	Stream<CdsAnnotation<?>> annotations();

	/**
	 * Returns an {@link Optional} wrapping a {@link CdsAnnotation}.
	 * 
	 * @param <T>  the expected type of the annotation value
	 * @param name the name of the {@code CdsAnnotation}
	 * @return an {@code Optional} describing the {@code CdsAnnotation} with the
	 *         given name, or an empty {@code Optional} if there is no
	 *         {@code CdsAnnotation} with this name
	 */
	<T> Optional<CdsAnnotation<T>> findAnnotation(String name);

	/**
	 * Returns a {@link Predicate} to filter {@code CdsAnnotatable(s)} that have a
	 * {@link CdsAnnotation} with the given name.
	 * 
	 * @param annotation the name of the {@code CdsAnnotation}
	 * @return a {@link Predicate} filtering by annotations
	 */
	static Predicate<CdsAnnotatable> byAnnotation(String annotation) {
		return a -> a.findAnnotation(annotation).isPresent();
	}

}
