/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.reflect;

import java.util.Optional;
import java.util.stream.Stream;

import com.sap.cds.ql.cqn.CqnPredicate;

public interface CdsAssociationType extends CdsType {

	/**
	 * Returns the target {@link CdsEntity} of this {@code CdsAssociation}.
	 * 
	 * @return the target {@code CdsEntity} of this association, not {@code null}
	 */
	CdsEntity getTarget();

	/**
	 * Returns the {@link Cardinality} of this {@code CdsAssociation}.
	 * 
	 * @return the {@code Cardinality} of this association, not {@code null}
	 */
	Cardinality getCardinality();

	/**
	 * Returns {@code true} if this is a composition.
	 * 
	 * @return {@code true} if this is a composition, otherwise {@code false}
	 */
	boolean isComposition();

	/**
	 * Returns a sequential {@code Stream} over the key {@link CdsElement}
	 * definitions of this association's target {@link CdsEntity}.
	 *
	 * @return a sequential {@code Stream} over the target's key {@code CdsElement}
	 *         definitions
	 */
	Stream<CdsElement> keys();

	/**
	 * Returns an {@link Optional} wrapping the on condition of this association.
	 * 
	 * @return an {@code Optional} describing the on condition of this association,
	 *         or an empty {@code Optional} if this association has no on condition
	 * @see CqnPredicate
	 */
	Optional<CqnPredicate> onCondition();

	@Override
	default boolean isAssociation() {
		return true;
	}

	@Override
	default String getQualifiedName() {
		return "cds.Association";
	}

	@Override
	default void accept(CdsVisitor visitor) {
		visitor.visit(this);
	}

	interface Cardinality {

		/**
		 * Returns the association's source cardinality, which can be '*' or a positive
		 * Integer.
		 * 
		 * @return the association's source cardinality, not {@code null}
		 */
		String getSourceMax();

		/**
		 * Returns the association's minimum target cardinality, which can be zero or a
		 * positive Integer.
		 * 
		 * @return the association's minimum target cardinality, not {@code null}
		 */
		String getTargetMin();

		/**
		 * Returns the association's maximum target cardinality, which can be '*' or a
		 * positive Integer.
		 * 
		 * @return the association's maximum target cardinality, not {@code null}
		 */
		String getTargetMax();

	}

}
