/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.reflect;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;

import com.sap.cds.CdsException;

public enum CdsBaseType {
	UUID("cds.UUID", String.class), BOOLEAN("cds.Boolean", Boolean.class), INTEGER("cds.Integer", Integer.class),
	INTEGER64("cds.Integer64", Long.class), DECIMAL("cds.Decimal", BigDecimal.class), @Deprecated
	DECIMAL_FLOAT("cds.DecimalFloat", BigDecimal.class), DOUBLE("cds.Double", Double.class),
	DATE("cds.Date", LocalDate.class), TIME("cds.Time", LocalTime.class), DATETIME("cds.DateTime", Instant.class),
	TIMESTAMP("cds.Timestamp", Instant.class), STRING("cds.String", String.class), BINARY("cds.Binary", byte[].class),
	LARGE_STRING("cds.LargeString", String.class), LARGE_BINARY("cds.LargeBinary", byte[].class);

	private final String cdsName;
	private final Class<?> javaType;

	private CdsBaseType(String cdsName, Class<?> javaType) {
		this.cdsName = cdsName;
		this.javaType = javaType;
	}

	public String cdsName() {
		return cdsName;
	}

	public Class<?> javaType() {
		return javaType;
	}

	@Override
	public String toString() {
		return cdsName;
	}

	public static CdsBaseType cdsType(String cdsType) {
		switch (cdsType) {
		case "cds.Boolean":
			return CdsBaseType.BOOLEAN;
		case "cds.UUID":
			return CdsBaseType.UUID;
		case "cds.Integer":
			return CdsBaseType.INTEGER;
		case "cds.Integer64":
			return CdsBaseType.INTEGER64;
		case "cds.Decimal":
			return CdsBaseType.DECIMAL;
		case "cds.DecimalFloat":
			return CdsBaseType.DECIMAL_FLOAT;
		case "cds.Double":
			return CdsBaseType.DOUBLE;
		case "cds.Date":
			return CdsBaseType.DATE;
		case "cds.Time":
			return CdsBaseType.TIME;
		case "cds.DateTime":
			return CdsBaseType.DATETIME;
		case "cds.Timestamp":
			return CdsBaseType.TIMESTAMP;
		case "cds.String":
			return CdsBaseType.STRING;
		case "cds.Binary":
			return CdsBaseType.BINARY;
		case "cds.LargeString":
			return CdsBaseType.LARGE_STRING;
		case "cds.LargeBinary":
			return CdsBaseType.LARGE_BINARY;
		default:
			throw new CdsException("Unknown CDS base type '" + cdsType + "'");
		}
	}

	public static Class<?> cdsJavaMediaType(CdsBaseType cdsType) {
		switch (cdsType) {
		case LARGE_STRING:
			return Reader.class;
		case LARGE_BINARY:
			return InputStream.class;
		default:
			throw new CdsException("@Core.MediaType is not supported for type " + cdsType);
		}
	}

}
