/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.reflect;

import java.util.function.Predicate;

public interface CdsDefinition extends CdsAnnotatable {

	/**
	 * Accepts a {@link CdsVisitor} visiting the elements of this definition
	 * (depth-first)
	 *
	 * @param visitor the {@code CdsVisitor}
	 */
	void accept(CdsVisitor visitor);

	/**
	 * Returns the fully qualified name of this definition.
	 *
	 * @return the fully qualified name of this definition
	 */
	String getQualifiedName();

	/**
	 * Returns the qualifier part of this definition's qualified name.
	 *
	 * @return the qualifier of the qualified name
	 */
	default String getQualifier() {
		String qualifiedName = getQualifiedName();
		int lastDotPos = qualifiedName.lastIndexOf('.');
		if (lastDotPos >= 0) {
			return qualifiedName.substring(0, lastDotPos);
		}
		return "";
	}

	/**
	 * Returns the unqualified name of this definition.
	 *
	 * @return the unqualified name
	 */
	default String getName() {
		String qualifiedName = getQualifiedName();
		int lastDotPos = qualifiedName.lastIndexOf('.');
		if (lastDotPos >= 0 && lastDotPos + 1 < qualifiedName.length()) {
			return qualifiedName.substring(lastDotPos + 1);
		}
		return qualifiedName;
	}

	/**
	 * Returns a {@link Predicate} to filter {@code CdsDefinition(s)} that are
	 * inside the given namespace. If the namespace is empty or {@code null}, all
	 * definitions match.
	 *
	 * @param namespace the namespace to filter, can be empty or {@code null}
	 * @return a {@link Predicate} filtering by namespace
	 */
	static Predicate<CdsDefinition> byNamespace(String namespace) {
		if (namespace == null || namespace.isEmpty()) {
			return d -> true;
		}
		return d -> d.getQualifiedName().startsWith(namespace + ".");
	}

}
