/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.reflect;

public interface CdsElement extends CdsAnnotatable {

	/**
	 * Returns the name of this {@code CdsElement}.
	 * 
	 * @return the namespace of this element, not {@code null}
	 */
	String getName();

	/**
	 * Returns the type of this {@code CdsElement}.
	 * 
	 * @param <T> the {@link CdsType} of this element
	 * @return the type of this element, not {@code null}
	 */
	<T extends CdsType> T getType();

	/**
	 * Returns the declaring type of this {@code CdsElement}.
	 * 
	 * @param <T> the type of the declaring type
	 * @return the declaring type of this element, not {@code null}
	 */
	<T extends CdsDefinition> T getDeclaringType();

	/**
	 * Returns {@code true} if this {@code CdsElement} is a key.
	 * 
	 * @return {@code true} if this element is a key, otherwise {@code false}
	 */
	boolean isKey();

	/**
	 * Returns {@code true} if this {@code CdsElement} is virtual.
	 * 
	 * @return {@code true} if this is a virtual element, otherwise {@code false}
	 */
	boolean isVirtual();

	/**
	 * Returns {@code true} if this {@code CdsElement} is unique.
	 * 
	 * @return {@code true} if this is a unique element, otherwise {@code false}
	 */
	boolean isUnique();

	/**
	 * Returns {@code true} if this {@code CdsElement} cannot be {@code null}.
	 * 
	 * @return {@code true} if this element cannot be {@code null}, otherwise
	 *         {@code false}
	 */
	boolean isNotNull();

	/**
	 * Accepts a {@link CdsVisitor} visiting this type.
	 * 
	 * @param visitor the {@code CdsVisitor}
	 */
	default void accept(CdsVisitor visitor) {
		visitor.visit(this);
	}

}
