/*******************************************************************
 * © 2023 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql.cqn.transformation;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;

/**
 * Transformation that determines top level instances that belong to a recursive
 * hierarchy.
 * 
 * The hierarchy is defined by the {@link #hierarchyReference() hierarchy
 * reference} and the {@link #hierarchyQualifier() hierarchy qualifier}.
 * 
 * The transformation takes an input set consisting of instances that belong to
 * the hierarchy. The output retains nodes of the top <i>n</i> {@link levels},
 * i.e. nodes with less then <i>n</i> ancestors.
 */
@Beta
public interface CqnTopLevelsTransformation extends CqnTransformation {

	/**
	 * Returns a reference, which identifies the entities that form a recursive
	 * hierarchy. The entity is annotated with {@code RecursiveHierarchy} with a
	 * qualifier whose value is given by the {@link #hierarchyQualifier()} method.
	 * 
	 * @return a reference, which identifies the entities that form a recursive
	 *         hierarchy
	 */
	CqnStructuredTypeRef hierarchyReference();

	/**
	 * Returns the qualifier of the {@code RecursiveHierarchy} annotation
	 * 
	 * @return the hierarchy qualifier
	 */
	String hierarchyQualifier();

	/**
	 * Returns a reference to the property that identifies the nodes of the
	 * hierarchy.
	 * 
	 * @return a reference to the node property
	 */
	CqnElementRef nodeProperty();

	/**
	 * Returns the number of levels, which the output set should contain
	 * 
	 * @return the number of levels or -1 if unlimited
	 */
	long levels();

	@Override
	default Kind kind() {
		return Kind.TOPLEVELS;
	}
}
