/************************************************************************
 * © 2021-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds;

/**
 * Provides runtime configuration parameters for {@link CdsDataStore}
 * implementations. The string constants in this class are configuration keys
 * consumed by implementations of this interface.
 */
public interface DataStoreConfiguration {

	// HANA
	String DOCSTORE_INTEGRATION_ENABLED = "cds.sql.hana.docstore";
	String IGNORE_LOCALE_ON_HANA = "cds.sql.hana.ignore-locale";
	String HANA_COMPATIBILITY_MODE = "cds.sql.hana.compatibilityMode";
	String HANA_COMPATIBILITY_MODE_LEGACY = "legacy"; // default
	String HANA_COMPATIBILITY_MODE_HEX = "hex";

	String COLLATE = "cds.sql.collate";
	String COLLATE_LOCALIZED = "localized-strings";

	String SUPPORTED_LOCALES = "supported_locales";

	String USE_LOCALIZED_VIEW_SEARCH_RESOLVER = "cds.sql.search.use-localized-view";

	String SELECT_STAR = "cds.sql.select.star";
	String SELECT_STAR_COLUMNS = "columns";

	String UNIVERSAL_CSN = "cds.model.universalCsn";

	// Don't log raw values of CQN queries by default
	String LOG_CQN_VALUES = "cds.ql.logging.log-values";

	String MAX_BATCH_SIZE = "cds.sql.max-batch-size";
	int MAX_BATCH_SIZE_DEFAULT = 1000;

	/**
	 * Get the property value for a given key.
	 * 
	 * @param key the key of the configuration property
	 * @return the value for the provided key or null
	 */
	String getProperty(String key);

	/**
	 * Get the property value for a given key. If no value is present for the key it
	 * returns the provided default value.
	 * 
	 * @param key          the key of the configuration property
	 * @param defaultValue the default value if no value is present for the provided
	 *                     key
	 * @return either the value for the provided key or the default value
	 */
	default String getProperty(String key, String defaultValue) {
		String strVal = getProperty(key);

		return strVal != null ? strVal : defaultValue;
	}

	/**
	 * Get the property value for a given key. If no value is present for the key it
	 * returns the provided default value.
	 * 
	 * @param key          the key of the configuration property
	 * @param defaultValue the default value if no value is present for the provided
	 *                     key
	 * @return either the value for the provided key or the default value
	 */
	default boolean getProperty(String key, boolean defaultValue) {
		String strVal = getProperty(key);

		return strVal != null ? Boolean.parseBoolean(strVal) : defaultValue;
	}

	/**
	 * Get the property value for a given key. If no value is present for the key it
	 * returns the provided default value.
	 * 
	 * @param key          the key of the configuration property
	 * @param defaultValue the default value if no value is present for the provided
	 *                     key
	 * @return either the value for the provided key or the default value
	 */
	default int getProperty(String key, int defaultValue) {
		String strVal = getProperty(key);

		return strVal != null ? Integer.valueOf(strVal) : defaultValue;
	}

	default boolean useLocaleSpecificView() {
		String ls = getProperty(SUPPORTED_LOCALES);

		return !(ls == null || ls.isEmpty() || "*".equals(ls));
	}

}
