/************************************************************************
 * © 2020-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds;

import static java.util.Arrays.stream;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;

/**
 * Builder class to programmatically create a {@link Result} to represent the
 * result of queries, insert, update and delete operations.
 */
public abstract class ResultBuilder {

	private static final Factory factory = Cds4jServiceLoader.load(Factory.class);

	private static ResultBuilder create() {
		return factory.create();
	}

	/**
	 * Creates a {@code ResultBuilder} to represent a query result.
	 * 
	 * @param rows the result rows
	 * @return the result builder
	 */
	public static ResultBuilder selectedRows(List<? extends Map<String, ?>> rows) {
		return create().rows(rows).rowCount(new long[] { rows.size() });
	}

	/**
	 * Creates a {@code ResultBuilder} for an INSERT or UPSERT result.
	 * 
	 * @param rows the rows in- or upserted
	 * @return the result builder
	 */
	public static ResultBuilder insertedRows(List<? extends Map<String, ?>> rows) {
		long[] rowCount = new long[rows.size()];
		Arrays.fill(rowCount, 1);
		return create().rows(rows).rowCount(rowCount);
	}

	/**
	 * Creates a {@code ResultBuilder} <span class="x x-first x-last">for a searched
	 * UPDATE,</span> <span class="x x-first x-last">updating</span> multiple rows
	 * with the same data.
	 * 
	 * @param updateCount the number of rows that were updated
	 * @param updateData  the update data
	 * @return the result builder
	 */
	public static ResultBuilder updatedRows(long updateCount, Map<String, ?> updateData) {
		return create().addUpdatedRows(updateCount, updateData);
	}

	/**
	 * Creates a {@code ResultBuilder} for a batch UPDATE result. The results of the
	 * individual UPDATE statements have to be added using
	 * {@link #addUpdatedRows(long, Map)}.
	 * 
	 * @return the result builder
	 */
	public static ResultBuilder batchUpdate() {
		return create();
	}

	/**
	 * Creates a {@code ResultBuilder} for a DELETE result.
	 * 
	 * @param deleteCount the number of rows that were deleted
	 * @return the result builder
	 */
	public static ResultBuilder deletedRows(long deleteCount) {
		return deletedRows(new long[] { deleteCount });
	}

	/**
	 * Creates a {@code ResultBuilder} for a batch DELETE result.
	 * 
	 * @param deleteCounts the number of rows deleted by the individual DELETE
	 *                     operations of the batch
	 * @return the result builder
	 */
	public static ResultBuilder deletedRows(int[] deleteCounts) {
		return create().rowCount(longArray(deleteCounts));
	}

	/**
	 * Creates a {@code ResultBuilder} for a batch DELETE result.
	 * 
	 * @param deleteCounts the number of rows deleted by the individual DELETE
	 *                     operations of the batch
	 * @return the result builder
	 */
	public static ResultBuilder deletedRows(long[] deleteCounts) {
		return create().rowCount(deleteCounts);
	}

	/**
	 * Specifies the source {@link CdsEntity} of this result.
	 * 
	 * @param entity the source entity
	 * @return this {@link ResultBuilder}
	 */
	public abstract ResultBuilder entity(CdsEntity entity);

	/**
	 * Specifies the {@link CdsStructuredType} that represents the row type of this
	 * result.
	 * 
	 * @param rowType the row type
	 * @return this {@link ResultBuilder}
	 */
	public abstract ResultBuilder rowType(CdsStructuredType rowType);

	/**
	 * Specifies the inline count of a query result.
	 * 
	 * @param inlineCount the inline count
	 * @return this {@link ResultBuilder}
	 */
	public abstract ResultBuilder inlineCount(long inlineCount);

	/**
	 * Adds an update result to this result builder that represents the result of a
	 * single UPDATE operation of a batch.
	 * 
	 * @param updateCount the number of rows that were updated
	 * @param updateData  the update data
	 * @return this {@link ResultBuilder}
	 */
	public abstract ResultBuilder addUpdatedRows(long updateCount, Map<String, ?> updateData);

	/**
	 * Adds multiple update results to this result builder that represent the result
	 * of the UPDATE operations of a batch.
	 * 
	 * @param updateCounts the number of rows that were updated per line of the
	 *                     batch
	 * @param updateData   the update data
	 * @return this {@link ResultBuilder}
	 */
	public abstract ResultBuilder addUpdatedRows(long[] updateCounts, List<? extends Map<String, ?>> updateData);

	/**
	 * Create a {@link Result} from this {@link ResultBuilder}
	 * 
	 * @return the result
	 */
	public abstract Result result();

	protected abstract ResultBuilder rows(Stream<? extends Map<String, ?>> rows);

	protected ResultBuilder rows(List<? extends Map<String, ?>> rows) {
		return rows(rows.stream());
	}

	protected abstract ResultBuilder rowCount(long[] rowCounts);

	private static long[] longArray(int[] updateCount) {
		return stream(updateCount).asLongStream().toArray();
	}

	public interface Factory {
		ResultBuilder create();
	}

}
