/************************************************************************
 * © 2020-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.ql;

import static java.util.Arrays.asList;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collector;

import com.sap.cds.ql.cqn.CqnArithmeticExpression;
import com.sap.cds.ql.cqn.CqnComparisonPredicate;
import com.sap.cds.ql.cqn.CqnConnectivePredicate;
import com.sap.cds.ql.cqn.CqnConnectivePredicate.Operator;
import com.sap.cds.ql.cqn.CqnContainmentTest;
import com.sap.cds.ql.cqn.CqnContainmentTest.Position;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnInPredicate;
import com.sap.cds.ql.cqn.CqnListValue;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnSortSpecification.Order;
import com.sap.cds.ql.cqn.CqnStar;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Modifier;

public interface CQL {

	BooleanValue TRUE = CDS.QL.create.booleanValue(true);
	BooleanValue FALSE = CDS.QL.create.booleanValue(false);
	NullValue NULL = CDS.QL.create.nullValue();

	/**
	 * Creates a {@link RefBuilder} from the given ref, which allows for in-place
	 * modification of the ref.
	 *
	 * @param ref the ref to be copied
	 * @return the ref builder
	 */
	static RefBuilder<StructuredTypeRef> copy(CqnStructuredTypeRef ref) {
		return CDS.QL.builder.copy(ref);
	}

	/**
	 * Creates a {@link RefBuilder} from the given ref, which allows for in-place
	 * modification of the ref.
	 *
	 * @param ref the element ref to be copied
	 * @return the ref builder
	 */
	static RefBuilder<ElementRef<?>> copy(CqnElementRef ref) {
		return CDS.QL.builder.copy(ref);
	}

	/**
	 * Creates a copy of the given expand.
	 *
	 * @param expand the expand to be copied
	 * @return the copy
	 */
	static Expand<?> copy(CqnExpand expand) {
		return CDS.QL.builder.copy(expand);
	}

	/**
	 * Creates a copy of the given predicate.
	 *
	 * @param pred the predicate to be copied
	 * @return the copy
	 */
	static Predicate copy(CqnPredicate pred) {
		return CDS.QL.builder.copy(pred);
	}

	/**
	 * Creates a modified copy of the given predicate.
	 *
	 * @param pred     the predicate to be copied
	 * @param modifier the modifier for altering the copy
	 * @return the modified copy
	 */
	static Predicate copy(CqnPredicate pred, Modifier modifier) {
		return CDS.QL.builder.copy(pred, modifier);
	}

	/**
	 * Creates a copy of the given CQN statement.
	 *
	 * @param <S>       the type of CqnStatement
	 * @param statement the CQN statement to be copied
	 * @return the copy
	 */
	static <S extends CqnStatement> S copy(S statement) {
		return CDS.QL.builder.copy(statement);
	}

	/**
	 * Creates a modified copy of the given CQN statement.
	 *
	 * @param <S>       the type of CqnStatement
	 * @param statement the CQN statement to be copied
	 * @param modifier  the modifier for altering the copy
	 * @return the modified copy
	 */
	static <S extends CqnStatement> S copy(S statement, Modifier modifier) {
		return CDS.QL.builder.copy(statement, modifier);
	}

	/**
	 * Creates a CDS QL function call that is send to the underlying data store.
	 *
	 * @param <T>  the return type of the function call
	 * @param name the name of the function to be called
	 * @param args the arguments of the function
	 * @return the CDS QL function call
	 */
	static <T> FunctionCall<T> func(String name, CqnValue... args) {
		return func(name, asList(args));
	}

	/**
	 * Creates a CDS QL function call that is send to the underlying data store.
	 *
	 * @param <T>  the return type of the function call
	 * @param name the name of the function to be called
	 * @param args the arguments of the function
	 * @return the CDS QL function call
	 */
	static <T> FunctionCall<T> func(String name, List<? extends CqnValue> args) {
		return CDS.QL.create.func(name, args);
	}

	/**
	 * Creates a CDS QL function call that is send to the underlying data store.
	 *
	 * @param name the name of the function to be called
	 * @param args the arguments of the function
	 * @return the CDS QL function call
	 */
	static BooleanFunction booleanFunc(String name, List<? extends CqnValue> args) {
		return CDS.QL.create.booleanFunc(name, args);
	}

	/**
	 * Creates a {@link Value} of type {@link T} with a hint that the value should
	 * be treated as constant.
	 *
	 * @param <T>   the type of the constant
	 * @param value the value
	 * @return the constant
	 */
	static <T> Literal<T> constant(T value) {
		return CDS.QL.create.constant(value);
	}

	/**
	 * Creates a {@link Value} of type {@link T}.
	 *
	 * @param <T>   the type of the non-constant value
	 * @param value the value of the non-constant value
	 * @return the value
	 * @see #param(String)
	 */
	static <T> Literal<T> val(T value) {
		return CDS.QL.create.val(value);
	}

	/**
	 * Negates a given {@link CqnPredicate}.
	 *
	 * @param predicate the {@code CqnPredicate} to be negated
	 * @return the negated predicate
	 */
	static Predicate not(CqnPredicate predicate) {
		return CDS.QL.create.not(predicate);
	}

	/**
	 * Creates a conjunction of two given predicates
	 *
	 * @param p1 1st operand of the conjunction
	 * @param p2 2nd operand of the conjunction
	 * @return the conjunction
	 */
	static Predicate and(CqnPredicate p1, CqnPredicate p2) {
		return and(asList(p1, p2));
	}

	/**
	 * Returns a {@link Collector} that connects the input predicates with AND.
	 *
	 * If there are none the {@link Collector} returns TRUE.
	 *
	 * @return the collector
	 */
	static Collector<CqnPredicate, ?, CqnPredicate> withAnd() {
		return CDS.QL.create.withAnd();
	}

	/**
	 * Creates a disjunction of two given predicates
	 *
	 * @param p1 1st operand of the disjunction
	 * @param p2 2nd operand of the disjunction
	 * @return the disjunction
	 */
	static Predicate or(CqnPredicate p1, CqnPredicate p2) {
		return or(asList(p1, p2));
	}

	/**
	 * Creates a conjunction of given predicates
	 *
	 * @param predicates the predicates to connect
	 * @return the conjunction
	 */
	static Predicate and(Iterable<? extends CqnPredicate> predicates) {
		return connect(Operator.AND, predicates);
	}

	/**
	 * Creates a disjunction of given predicates
	 *
	 * @param predicates the predicates to connect
	 * @return the disjunction
	 */
	static Predicate or(Iterable<? extends CqnPredicate> predicates) {
		return connect(Operator.OR, predicates);
	}

	/**
	 * Returns a {@link Collector} that connects the input predicates with OR.
	 *
	 * If there are none the {@link Collector} returns FALSE.
	 *
	 * @return the collector
	 */
	static Collector<CqnPredicate, ?, CqnPredicate> withOr() {
		return CDS.QL.create.withOr();
	}

	/**
	 * Creates a logical connection of given predicates {@link CqnPredicate}.
	 *
	 * @param operator   operator to connect the predicates with
	 * @param predicates the predicates to connect
	 * @return the logical connection
	 */
	static Predicate connect(CqnConnectivePredicate.Operator operator, Iterable<? extends CqnPredicate> predicates) {
		return CDS.QL.create.connect(operator, predicates);
	}

	/**
	 * Creates an indexed {@link Parameter} to be used in CQN statements. Can be
	 * used with positional parameter values.
	 *
	 * @param <T>   the type of the parameter
	 * @param index the parameter index. If used with positional parameter values,
	 *              start with 0 for the first parameter
	 * @return the parameter
	 */
	static <T> Parameter<T> param(int index) {
		return param(String.valueOf(index));
	}

	/**
	 * Creates a named {@link Parameter} to be used in queries.
	 *
	 * @param <T>  the type of the parameter
	 * @param name the parameter name
	 * @return the parameter
	 */
	static <T> Parameter<T> param(String name) {
		return CDS.QL.create.param(name);
	}

	/**
	 * Creates a CQN plain value, which is directly send to the underlying data
	 * store.
	 *
	 * @param value the value content
	 * @return the plain value
	 */
	static Value<?> plain(String value) {
		return CDS.QL.create.plain(value);
	}

	/**
	 * Creates a count all function call.
	 *
	 * @return the count value
	 */
	static Value<Long> count() {
		return func("COUNT");
	}

	/**
	 * Creates a count function call on the given element.
	 *
	 * @param element the ref to the element to be counted
	 * @return the count value
	 */
	static Value<Long> count(CqnElementRef element) {
		return func("COUNT", element);
	}

	/**
	 * Creates a min function call on the given element or expression.
	 *
	 * @param <T>   the type of the value
	 * @param value the ref to the element to be counted or an expression
	 * @return the minimum value
	 */
	static <T> Value<T> min(CqnValue value) {
		return func("min", value);
	}

	/**
	 * Creates a max function call on the given element or expression.
	 *
	 * @param <T>   the type of the value
	 * @param value the ref to the element to be counted or an expression
	 * @return the maximum value
	 */
	static <T> Value<T> max(CqnValue value) {
		return func("max", value);
	}

	/**
	 * Creates a sum function call on the given element or expression.
	 *
	 * @param value the ref to the element to be counted or an expression
	 * @return the sum value
	 */
	static Value<Number> sum(CqnValue value) {
		return func("sum", value);
	}

	/**
	 * Creates an average function call on the given element or expression.
	 *
	 * @param value the ref to the element to be counted or an expression
	 * @return the average value
	 */
	static Value<Number> average(CqnValue value) {
		return func("avg", value);
	}

	/**
	 * Creates an count distinct function call on the given element or expression.
	 *
	 * @param value the ref to the element to be counted or an expression
	 * @return the count of distinct values
	 */
	static Value<Long> countDistinct(CqnValue value) {
		return CDS.QL.create.countDistinct(value);
	}

	/**
	 * Creates a predicate that tests whether a string value contains a given
	 * substring
	 *
	 * @param value           the string value
	 * @param substring       the substring to test for
	 * @param caseInsensitive whether the test should be case insensitive
	 * @return the predicate for the containment test
	 */
	static Predicate contains(CqnValue value, CqnValue substring, boolean caseInsensitive) {
		return CDS.QL.create.containment(Position.ANY, value, substring, caseInsensitive);
	}

	/**
	 * Creates a predicate that tests whether a string value starts with a given
	 * prefix
	 *
	 * @param value           the string value
	 * @param prefix          the prefix to test for
	 * @param caseInsensitive whether the test should be case insensitive
	 * @return the predicate for the containment test
	 */
	static Predicate startsWith(CqnValue value, CqnValue prefix, boolean caseInsensitive) {
		return CDS.QL.create.containment(Position.START, value, prefix, caseInsensitive);
	}

	/**
	 * Creates a predicate that tests whether a string value ends with a given
	 * suffix
	 *
	 * @param value           the string value
	 * @param suffix          the suffix to test for
	 * @param caseInsensitive whether the test should be case insensitive
	 * @return the predicate for the containment test
	 */
	static Predicate endsWith(CqnValue value, CqnValue suffix, boolean caseInsensitive) {
		return CDS.QL.create.containment(Position.END, value, suffix, caseInsensitive);
	}

	/**
	 * Creates a predicate that tests whether a string value contains a given
	 * substring at a given position {@link CqnContainmentTest.Position}
	 *
	 * @param position        the position
	 * @param value           the string value
	 * @param substring       the substring to test for
	 * @param caseInsensitive whether the test should be case insensitive
	 * @return the predicate for the containment test
	 */
	static Predicate containment(CqnContainmentTest.Position position, CqnValue value, CqnValue substring,
			boolean caseInsensitive) {
		return CDS.QL.create.containment(position, value, substring, caseInsensitive);
	}

	/**
	 * Creates a CDS QL expression for converting the given string value to lower
	 * case using the rules of the underlying data store.
	 *
	 * @param val the String value to be converted
	 * @return the CDS QL expression for lower case conversion
	 */
	static Value<String> toLower(Value<String> val) {
		return CDS.QL.create.toLower(val);
	}

	/**
	 * Creates a CDS QL expression for converting the given string value to lower
	 * case using the rules of the underlying data store.
	 *
	 * @param val the String value to be converted
	 * @return the CDS QL expression for lower case conversion
	 */
	static Value<String> toLower(String val) {
		return CDS.QL.create.toLower(val(val));
	}

	/**
	 * Creates a CDS QL expression for converting the given string value to upper
	 * case using the rules of the underlying data store.
	 *
	 * @param val the String value to be converted
	 * @return the CDS QL expression for upper case conversion
	 */
	static Value<String> toUpper(Value<String> val) {
		return CDS.QL.create.toUpper(val);
	}

	/**
	 * Creates a CDS QL expression for converting the given string value to upper
	 * case using the rules of the underlying data store.
	 *
	 * @param val the String value to be converted
	 * @return the CDS QL expression for upper case conversion
	 */
	static Value<String> toUpper(String val) {
		return CDS.QL.create.toUpper(val(val));
	}

	/**
	 * Creates a {@link StructuredType} representing a CDS entity.
	 *
	 * @param qualifiedName the qualified name of the CDS entity
	 * @return the {@code StructuredType} representing the CDS entity
	 */
	static StructuredType<?> entity(String qualifiedName) {
		return CDS.QL.create.entity(qualifiedName);
	}

	/**
	 * Creates a {@link StructuredType} representing a CDS entity.
	 *
	 * @param <T>  the type of the entity
	 * @param type the class of the entity type
	 * @return the {@code StructuredType} representing the CDS entity
	 */
	static <T extends StructuredType<T>> T entity(Class<T> type) {
		return CDS.QL.create.entity(type);
	}

	/**
	 * Creates a {@link StructuredType} representing a path from a CDS entity to
	 * another entity.
	 *
	 * @param path a String of association element names separated by '.'
	 * @return the {@code StructuredType} representing the path to a CDS entity
	 */
	static StructuredType<?> to(String path) {
		return CDS.QL.create.to(path);
	}

	/**
	 * Creates a {@link StructuredType} representing a path from a CDS entity to
	 * another entity.
	 * 
	 * To create a {@link StructuredTypeRef}, use {@link StructuredType#asRef()
	 * CQL.to(segments).asRef()}.
	 *
	 * @param segments the segments of the path
	 * @return the {@code StructuredType} representing the path to a CDS entity
	 */
	static StructuredType<?> to(List<? extends Segment> segments) {
		return CDS.QL.create.to(segments);
	}

	/**
	 * Creates an {@link ElementRef} representing a path from a CDS entity to an
	 * element of this or another entity.
	 *
	 * @param <T>  the type of the element
	 * @param path a String of element names separated by '.', where all except the
	 *             last segment represent association elements.
	 * @return the {@code ElementRef} representing the path to the CDS element
	 */
	static <T> ElementRef<T> get(String path) {
		return CDS.QL.create.get(path);
	}

	/**
	 * Creates an {@link ElementRef} representing a path from a CDS entity to an
	 * element of this or another entity.
	 *
	 * @param <T>      the type of the element
	 * @param segments the segments of the path
	 * @return the {@code ElementRef} representing the path to the CDS element
	 */
	static <T> ElementRef<T> get(List<? extends Segment> segments) {
		return CDS.QL.create.get(segments);
	}

	/**
	 * Creates a segment of a structured type or element reference without filter.
	 * 
	 * Use {@link #refSegment(String, CqnPredicate)} to create a segment with filter
	 * condition.
	 *
	 * @param id the id of the segment
	 * @return the segment
	 */
	static Segment refSegment(String id) {
		return CDS.QL.create.refSegment(id);
	}

	/**
	 * Creates a segment of a structured type or element reference with a filter
	 * condition.
	 *
	 * @param id     the id of the segment
	 * @param filter the filter of the segment
	 * @return the segment
	 */
	static Segment refSegment(String id, CqnPredicate filter) {
		return CDS.QL.create.refSegment(id, filter);
	}

	/**
	 * Creates a list of reference segments without filter.
	 *
	 * @param segmentIds the ids of the segments
	 * @return the segments
	 */
	static List<Segment> refSegments(List<String> segmentIds) {
		return CDS.QL.create.refSegments(segmentIds);
	}

	/**
	 * Creates an arithmetic expression.
	 *
	 * @param left  the left-hand side of the expression
	 * @param op    the operator
	 * @param right the right-hand side of the expression
	 * @return the arithmetic expression
	 */
	static Value<Number> expression(CqnValue left, CqnArithmeticExpression.Operator op, CqnValue right) {
		return CDS.QL.create.expression(left, op, right);
	}

	/**
	 * Creates a comparison predicate.
	 *
	 * @param lhs the left-hand side of the comparison
	 * @param op  the comparison operator
	 * @param rhs the right-hand side of the comparison
	 * @return the comparison predicate
	 */
	static Predicate comparison(CqnValue lhs, CqnComparisonPredicate.Operator op, CqnValue rhs) {
		return CDS.QL.create.comparison(lhs, op, rhs);
	}

	/**
	 * Creates an {@link CqnInPredicate in} predicate.
	 *
	 * @param value  the value to be checked if it's contained in the collection
	 * @param values the collection of values
	 * @return the {@code in} predicate
	 */
	static Predicate in(CqnValue value, Collection<? extends CqnValue> values) {
		return CDS.QL.create.in(value, values);
	}

	/**
	 * Creates an {@link CqnInPredicate in} predicate.
	 *
	 * @param value    the value to be checked if it's contained in the collection
	 * @param valueSet the value describing a collection of values
	 * @return the {@code in} predicate
	 */
	static Predicate in(CqnValue value, CqnValue valueSet) {
		return CDS.QL.create.in(value, valueSet);
	}

	/**
	 * Creates a predicate that checks if the values of the given elements match any
	 * of the given value sets.
	 * 
	 * The value sets to test against are computed from the given value maps by
	 * extracting the values for the given elements.
	 *
	 * @param elements  the element names
	 * @param valueMaps the value maps to match against
	 */
	static Predicate in(List<String> elements, Collection<? extends Map<String, ?>> valueMaps) {
		return CDS.QL.create.in(elements, valueMaps);
	}

	/**
	 * Creates a search predicate.
	 *
	 * @param term the search term
	 * @return the search predicate
	 */
	static Predicate search(String term) {
		return CDS.QL.create.search(term);
	}

	/**
	 * Creates an exists predicate that tests if a given subquery returns any row
	 *
	 * @param subQuery the subquery that performs the existence test
	 * @return the exists predicate
	 */
	static Predicate exists(CqnSelect subQuery) {
		return CDS.QL.create.exists(subQuery);
	}

	/**
	 * Creates a match predicate that tests if a given ref matches a given filter
	 *
	 * @param ref        the reference
	 * @param pred       the filter
	 * @param quantifier all or any
	 * @return the match predicate
	 */
	static Predicate match(CqnStructuredTypeRef ref, CqnPredicate pred, CqnMatchPredicate.Quantifier quantifier) {
		return CDS.QL.create.match(ref, pred, quantifier);
	}

	/**
	 * Creates an {@link ElementRef} to be used in queries to represent $now.
	 *
	 * @return the now value
	 */
	static Value<Instant> now() {
		return CDS.QL.create.now().type(Instant.class);
	}

	/**
	 * Creates an {@link ElementRef} on the available $validFrom value.
	 *
	 * @return the validFrom value
	 */
	static Value<Instant> validFrom() {
		return CDS.QL.create.validFrom().type(Instant.class);
	}

	/**
	 * Creates an {@link ElementRef} on the available $validTo value.
	 *
	 * @return the validTo value
	 */
	static Value<Instant> validTo() {
		return CDS.QL.create.validTo().type(Instant.class);
	}

	/**
	 * Creates an {@link ElementRef} on the available $user.locale value.
	 *
	 * @return the locale string value
	 */
	static Value<String> userLocale() {
		return CDS.QL.create.userLocale().type(String.class);
	}

	/**
	 * Creates an {@link ElementRef} on the available $user.id value.
	 *
	 * @return the user.id value
	 */
	static Value<String> userId() {
		return CDS.QL.create.userId().type(String.class);
	}

	static CqnStar star() {
		return CqnStar.star();
	}

	/**
	 * Creates a {@link CqnListValue} composing given values
	 *
	 * @param values the components
	 * @return the list value
	 */
	static CqnListValue list(List<? extends CqnValue> values) {
		return CDS.QL.create.list(values);
	}

	/**
	 * Creates a {@link CqnListValue} composing given values
	 *
	 * @param values the items
	 * @return the list value
	 */
	static CqnListValue list(CqnValue... values) {
		return list(Arrays.asList(values));
	}

	/**
	 * Creates a predicate from a map of element names to values or parameters. The
	 * map entries are transformed into equality predicates and joined via
	 * <b>and</b>.
	 *
	 * @param elementToValueMap the element name to value map defining the condition
	 * @return the matching predicate
	 */
	static Predicate matching(Map<String, ?> elementToValueMap) {
		return CDS.QL.builder.matching(elementToValueMap);
	}

	/**
	 * Creates a {@link CqnSortSpecification} that sorts a value by a given order
	 *
	 * @param value the value
	 * @param order the order
	 * @return the sort specification
	 */
	static CqnSortSpecification sort(CqnValue value, Order order) {
		return CDS.QL.create.sort(value, order);
	}

}
