/************************************************************************
 * © 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.ql;

import java.util.Map;
import java.util.function.Function;
import java.util.function.UnaryOperator;

import com.sap.cds.CdsDataStore;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.reflect.CdsEntity;

/**
 * The Upsert builder allows to construct a CDS QL Upsert statements, which can
 * be executed via the {@link CdsDataStore}.
 * 
 * An Upsert statement updates an entity from the given data. The key of the entity
 * that is upserted is expected to be in the data. If the entity doesn't exist and the key 
 * is given in the data, it inserts a new entry.
 */
public interface Upsert extends CqnUpsert, Statement<Upsert> {

	/**
	 * Creates an upsert statement to upsert entries into a specified entity set.
	 *
	 * @param qualifiedName the fully qualified name of the CDS entity set
	 * @return the upsert statement
	 */
	static Upsert into(String qualifiedName) {
		return into(qualifiedName, e -> e);
	}

	/**
	 * Creates an upsert statement to upsert entries into a specified entity set.
	 *
	 * @param ref the ref to the entity
	 * @return the upsert statement
	 */
	static Upsert into(CqnStructuredTypeRef ref) {
		return CDS.QL.builder.upsert(ref);
	}

	/**
	 * Creates an upsert statement to upsert entries into a specified entity set.
	 *
	 * @param <T>    the type of the entity set
	 * @param entity the structured type representing the entity set
	 * @return the upsert statement
	 */
	public static <T extends StructuredType<?>> Upsert into(T entity) {
		return CDS.QL.builder.upsert(entity);
	}

	/**
	 * Creates an upsert statement to upsert entries into a specified entity set.
	 *
	 * @param qualifiedName the fully qualified name of the CDS entity set
	 * @param path          a path expression navigating from the root entity to the
	 *                      target entity of the upsert statement
	 * @return the upsert statement
	 */
	static Upsert into(String qualifiedName, UnaryOperator<StructuredType<?>> path) {
		return CDS.QL.builder.upsert(qualifiedName, path);
	}

	/**
	 * Creates an upsert statement to upsert entries into a specified entity set.
	 *
	 * @param <T>    the type of the entity set
	 * @param entity the static model representation of the entity set
	 * @return the upsert statement
	 */
	static <T extends StructuredType<T>> Upsert into(Class<T> entity) {
		return into(entity, e -> e);
	}

	/**
	 * Creates an upsert statement to upsert entries into a specified entity set.
	 *
	 * @param <R>    the type of the root entity
	 * @param <T>    the type of the target entity
	 * @param entity the static model representation of the entity set
	 * @param path   a path expression navigating from the root entity to the target
	 *               entity of the upsert statement
	 * @return the upsert statement
	 */
	static <R extends StructuredType<R>, T extends StructuredType<T>> Upsert into(Class<R> entity,
			Function<R, T> path) {
		return CDS.QL.builder.upsert(entity, path);
	}

	/**
	 * Creates an upsert statement to upsert entries into a specified entity set.
	 *
	 * @param entity the model representation of the entity set obtained by
	 *               reflection
	 * @return the upsert statement
	 * @see com.sap.cds.reflect.CdsModel#findEntity(String)
	 * @see com.sap.cds.reflect.CdsModel#entities()
	 */
	static Upsert into(CdsEntity entity) {
		return into(entity, e -> e);
	}

	/**
	 * Creates an upsert statement to upsert entries into a specified entity set.
	 *
	 * @param entity the model representation of the entity set obtained by
	 *               reflection
	 * @param path   a path expression navigating from the root entity to the target
	 *               entity of the upsert statement
	 * @return the upsert statement
	 * @see com.sap.cds.reflect.CdsModel#findEntity(String)
	 * @see com.sap.cds.reflect.CdsModel#entities()
	 */
	static Upsert into(CdsEntity entity, UnaryOperator<StructuredType<?>> path) {
		return CDS.QL.builder.upsert(entity, path);
	}

	/**
	 * Copies the given {@link CqnUpsert} into an {@link Upsert} builder.
	 *
	 * @param upsert the {@code CqnUpsert} to be copied
	 * @return the modifiable upsert statement copy
	 */
	static Upsert copy(CqnUpsert upsert) {
		return CDS.QL.builder.copy(upsert);
	}

	/**
	 * Creates an upsert statement to upsert data for a specified entity set.
	 *
	 * @param cqnUpsert the CQN representation of the upsert statement
	 * @return the upsert statement
	 */
	static Upsert cqn(String cqnUpsert) {
		return CDS.QL.parser.upsert(cqnUpsert);
	}

	/**
	 * Sets the batch data to be inserted or updated by the upsert statement. The
	 * data is given as a collection of maps from element names of the target entity
	 * set to their new values. The data must contain the entities' keys.
	 *
	 * @param entries a collection of data where every element is the canonical
	 *                representation
	 * @return the upsert statement
	 */
	Upsert entries(Iterable<? extends Map<String, ?>> entries);

	/**
	 * Adds a single entry to be upserted into the entity set. The data is given as
	 * a {@literal Map<String, Object>} that maps element names of the target entity
	 * set to the values to be upserted.
	 * <p>
	 * The value can be deeply structured to represent a structured document:
	 * <ul>
	 * <li>for single-valued relationships the value of the element is of type
	 * {@literal Map<String, Object>}</li>
	 * <li>for collection-valued relationships the value of the element is of type
	 * {@literal List<Map<String, Object>>}</li>
	 * </ul>
	 *
	 * @param entry the data to be upserted as a map
	 *
	 * @return the upsert statement
	 */
	Upsert entry(Map<String, ?> entry);

	/**
	 * Adds a single entry to be upserted into the entity set.
	 * <p>
	 * The value can be deeply structured to represent a structured document:
	 * <ul>
	 * <li>for single-valued relationships the value of the element is of type
	 * {@literal Map<String, Object>}</li>
	 * <li>for collection-valued relationships the value of the element is of type
	 * {@literal List<Map<String, Object>>}</li>
	 * </ul>
	 *
	 * @param elementName the element name of the target entity for the value to be
	 *                    upserted
	 * @param value       the data to be upserted for the element
	 *
	 * @return the upsert statement
	 */
	Upsert entry(String elementName, Object value);

}
