/************************************************************************
 * © 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.ql.cqn;

import java.util.List;
import java.util.Optional;

public interface CqnSelect extends CqnFilterableStatement, CqnEntitySelector, CqnSource {

	/**
	 * The from clause of this select statement.
	 * 
	 * @return a reference to this statement's {@link CqnSource}
	 */
	CqnSource from();

	/**
	 * @return whether statement is a SELECT DISTINCT
	 */
	boolean isDistinct();

	/**
	 * @deprecated instead use {@link CqnEntitySelector#items}
	 * @return the select list items
	 */
	@Deprecated
	default List<CqnSelectListItem> columns() {
		return items();
	}

	List<CqnValue> groupBy();

	List<String> excluding();

	Optional<CqnPredicate> having();

	Optional<CqnLock> getLock();

	Optional<CqnPredicate> search();

	@Override
	default boolean isSelect() {
		return true;
	}

	@Override
	default CqnSelect asSelect() {
		return this;
	}

	/**
	 * Traverses the clauses of this {@code CqnSelect} with a given {@code visitor}.
	 *
	 * If the source of this select is a structured type reference it is visited
	 * first.
	 * 
	 * The other clauses are traversed depth-first in the following order:
	 * 
	 * <ul>
	 * <li>items</li>
	 * <li>where</li>
	 * <li>search</li>
	 * <li>group by</li>
	 * <li>having</li>
	 * <li>order by</li>
	 * </ul>
	 * 
	 * Afterwards this {@code CqnSelect} is passed to the
	 * {@link CqnVisitor#visit(CqnSelect)} method.
	 * 
	 * @param visitor the {@link CqnVisitor}
	 */
	default void accept(CqnVisitor visitor) {
		if (from().isRef()) {
			ref().accept(visitor);
		}

		dispatch(visitor);

		visitor.visit(this);
	}

	default void dispatch(CqnVisitor visitor) {
		items().forEach(c -> c.accept(visitor));
		where().ifPresent(w -> w.accept(visitor));
		search().ifPresent(s -> s.accept(visitor));
		groupBy().forEach(c -> c.accept(visitor));
		having().ifPresent(h -> h.accept(visitor));
		orderBy().forEach(o -> o.accept(visitor));
	}
}
