/************************************************************************
 * © 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.reflect;

import java.util.Optional;
import java.util.stream.Stream;

import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnPredicate;

public interface CdsAssociationType extends CdsType {

	/**
	 * Returns the target {@link CdsEntity} of this {@code CdsAssociation}.
	 * 
	 * @return the target {@code CdsEntity} of this association, not {@code null}
	 */
	CdsEntity getTarget();

	/**
	 * Returns an {@link Optional} wrapping the target Aspect's {@link CdsType}
	 * provided this is a composition that targets an aspect.
	 * 
	 * @return {@code Optional} holding the target aspect's {@code CdsType} if this
	 *         is a composition of aspects, or an empty {@code Optional} otherwise
	 */
	Optional<CdsType> getTargetAspect();

	/**
	 * Returns the {@link Cardinality} of this {@code CdsAssociation}.
	 * 
	 * @return the {@code Cardinality} of this association, not {@code null}
	 */
	Cardinality getCardinality();

	/**
	 * Returns {@code true} if this is a composition.
	 * 
	 * @return {@code true} if this is a composition, otherwise {@code false}
	 */
	boolean isComposition();

	/**
	 * Returns a sequential {@code Stream} of {@link CqnElementRef element refs}
	 * that are used to identify the {@link #getTarget target entity} of this
	 * association.
	 * 
	 * This is only supported for managed to-one associations.
	 * 
	 * @return a {@code Stream} of the referenced target entity elements, or an
	 *         empty stream for unmanaged and to-many associations.
	 */
	Stream<CqnElementRef> refs();

	/**
	 * Returns an {@link Optional} wrapping the on condition of this association.
	 * 
	 * @return an {@code Optional} describing the on condition of this association,
	 *         or an empty {@code Optional} if this association has no on condition
	 * @see CqnPredicate
	 */
	Optional<CqnPredicate> onCondition();

	@Override
	default boolean isAssociation() {
		return true;
	}

	@Override
	default String getQualifiedName() {
		return "cds.Association";
	}

	@Override
	default void accept(CdsVisitor visitor) {
		visitor.visit(this);
	}

	interface Cardinality {

		/**
		 * Returns the association's source cardinality, which can be '*' or a positive
		 * Integer.
		 * 
		 * @return the association's source cardinality, not {@code null}
		 */
		String getSourceMax();

		/**
		 * Returns the association's minimum target cardinality, which can be zero or a
		 * positive Integer.
		 * 
		 * @return the association's minimum target cardinality, not {@code null}
		 */
		String getTargetMin();

		/**
		 * Returns the association's maximum target cardinality, which can be '*' or a
		 * positive Integer.
		 * 
		 * @return the association's maximum target cardinality, not {@code null}
		 */
		String getTargetMax();

	}

}
