/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import com.sap.cds.ql.Select;
import com.sap.cds.reflect.CdsStructuredType;

/**
 * Represents the result data returned by the {@link CdsDataStore}.
 * {@code Result} can be serialized to JSON and accessed as an {@code Iterable}
 * of {@code Map} or in a typed way via an accessor interface.
 */
public interface Result extends Iterable<Row>, JSONizable {

	/**
	 * Returns the single row.
	 * 
	 * @return a {@code Row}
	 * 
	 * @throws EmptyResultException     if the result is empty
	 * @throws NonUniqueResultException if the result contains more than one row
	 */
	Row single();

	/**
	 * Returns the single row mapped to the given {@code type}.
	 * 
	 * @param <T>  the return type
	 * @param type the type the row is mapped to
	 * @return a row mapped to {@code T}
	 * 
	 * @throws EmptyResultException     if the result is empty
	 * @throws NonUniqueResultException if the result contains more than one row
	 */
	<T> T single(Class<T> type);

	/**
	 * Returns the first row.
	 * 
	 * @return an optional {@code Row}
	 */
	Optional<Row> first();

	/**
	 * Returns the first row mapped to the given {@code type}.
	 * 
	 * @param <T>  the type of the {@code Optional}
	 * @param type the type the row is mapped to
	 * @return an {@code Optional} row mapped to {@code T}
	 */
	<T> Optional<T> first(Class<T> type);

	/**
	 * Returns a sequential {@code Stream} of the result rows.
	 *
	 * @return a {@code Stream} of rows
	 */
	Stream<Row> stream();

	/**
	 * Returns a sequential {@code Stream} of the result rows mapped to the given
	 * {@code type}.
	 *
	 * @param <T>  the type of the {@code Stream}
	 * @param type the type the result rows are mapped to
	 * @return a {@code Stream} of the given {@code type}
	 */
	<T> Stream<T> streamOf(Class<T> type);

	/**
	 * Returns a list of the result rows.
	 *
	 * @return a list of {@code Row}
	 */
	List<Row> list();

	/**
	 * Returns a list of the result rows mapped to the given {@code type}.
	 *
	 * @param <T>  the type of the {@code List}
	 * @param type the type the result rows are mapped to
	 * @return a list of the given {@code type}
	 */
	<T> List<T> listOf(Class<T> type);

	/**
	 * Returns the row type of this result.
	 *
	 * @return the row type
	 */
	CdsStructuredType rowType();

	/**
	 * Returns the number of rows affected or returned by the query. In case of a
	 * batch query the sum over all batch row counts.
	 *
	 * @return the row count, -1 if undefined
	 */
	long rowCount();

	/**
	 * Returns the number of rows affected or returned by the query with the given
	 * batch index.
	 *
	 * @param batch the batch index
	 * @return the row count of the batch, -1 if undefined
	 * 
	 * @throws IndexOutOfBoundsException if the batch index is out of range
	 */
	long rowCount(int batch);

	/**
	 * Returns the number of batches.
	 *
	 * @return the number of batches, -1 if undefined
	 */
	int batchCount();

	/**
	 * Returns the total number of root entities matching the query as requested by
	 * {@link Select#inlineCount}.
	 * 
	 * @return the row count, -1 if undefined
	 */
	long inlineCount();

}
