/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql.cqn;

import java.util.List;

import com.sap.cds.CdsException;

/**
 * The entity selector describes a basic selection from an entity set.
 */
public interface CqnEntitySelector {

	/**
	 * Returns the reference to the entity set of this selector.
	 * 
	 * @throws CdsException if this selector does not have a reference
	 * @return the reference
	 */
	CqnStructuredTypeRef ref();

	/**
	 * Returns the selected items
	 * 
	 * @return the select list items
	 */
	List<CqnSelectListItem> items();

	/**
	 * Returns the order of the selection result
	 * 
	 * @return the result order
	 */
	List<CqnSortSpecification> orderBy();

	/**
	 * Dispatches a given visitor to all clause except the ref or source of this
	 * entity selector
	 * 
	 * @param visitor the visitor
	 */
	default void dispatch(CqnVisitor visitor) {
		items().forEach(c -> c.accept(visitor));
		orderBy().forEach(o -> o.accept(visitor));
	}

	/**
	 * Returns the maximum number of rows to be returned
	 * 
	 * @return the maximum number of rows to be returned or -1 if unlimited
	 */
	long top();

	/**
	 * Returns the number of rows to be skipped
	 * 
	 * @return the number of rows to be skipped
	 */
	long skip();

	/**
	 * Returns whether this entity selector includes an inline count.
	 *
	 * @return true if this entity selector includes an inline count, otherwise
	 *         false
	 */
	boolean hasInlineCount();

	/**
	 * Returns whether this entity selector has a top or skip
	 * 
	 * @return {@code true} if there is a top or skip
	 */
	default boolean hasLimit() {
		return top() >= 0 || skip() > 0;
	}
}
