/*
 * © 2018-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds;

import static java.util.Collections.emptyMap;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.Upsert;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.reflect.CdsEntity;
import java.util.Map;
import java.util.stream.Stream;

/**
 * The CdsDataStore is used to interact with a data source. It allows to execute CDS QL {@link
 * Select queries}, as well as {@link Insert insert}, {@link Upsert upsert}, {@link Update update}
 * and {@link Delete delete} statements.
 */
public interface CdsDataStore {

  /**
   * Executes a {@link CqnSelect} statement with optional positional values for indexed parameters,
   * see {@link CQL#param(int)}.
   *
   * @param select the CQN select statement to be executed
   * @param paramValues the positional parameter values
   * @return the {@link Result} of the query
   * @throws CdsLockTimeoutException if the statement uses {@link Select#lock() locking} but a lock
   *     can't be acquired
   */
  Result execute(CqnSelect select, Object... paramValues);

  /**
   * Executes a {@link CqnSelect} statement with values for named parameters, see {@link
   * CQL#param(String)}.
   *
   * @param select the CQN select statement to be executed
   * @param namedValues the named parameter values
   * @return the {@link Result} of the query
   * @throws CdsLockTimeoutException if the statement uses {@link Select#lock() locking} but a lock
   *     can't acquired
   */
  Result execute(CqnSelect select, Map<String, Object> namedValues);

  /**
   * Executes a {@link CqnSelect} statement as batch with the given named parameter values. Named
   * parameters are required either in the where clause, or in one infix filter of the from clause.
   *
   * <p>If the number of entries in valueSets is larger than the default batch size, the select
   * statement is executed multiple times and the result sets are combined, in this case order by
   * clauses are not supported and an exception is thrown.
   *
   * @param select the CQN select statement to be executed
   * @param valueSets the named parameter value sets
   * @return the {@link Result} of the batch select
   * @throws CdsLockTimeoutException if the statement uses {@link Select#lock() locking} but a lock
   *     can't acquired
   */
  @Beta
  Result execute(CqnSelect select, Iterable<Map<String, Object>> valueSets);

  /**
   * Executes a {@link CqnSelect} statement as batch with the given named parameter values. Named
   * parameters are required either in the where clause, or in one infix filter of the from clause.
   *
   * <p>If the number of entries in valueSets is larger than maxBatchSize, the select statement is
   * executed multiple times and the result sets are combined, in this case order by clauses are not
   * supported and an exception is thrown.
   *
   * @param select the CQN select statement to be executed
   * @param valueSets the named parameter value sets
   * @param maxBatchSize the maximum batch size
   * @return the {@link Result} of the batch select
   * @throws CdsLockTimeoutException if the query uses {@link Select#lock() locking} but a lock
   *     can't acquired
   */
  @Beta
  Result execute(CqnSelect select, Iterable<Map<String, Object>> valueSets, int maxBatchSize);

  /**
   * Executes a {@link CqnInsert} statement.
   *
   * @param insert the CQN insert statement to be executed
   * @return the {@link Result} of the insert
   */
  Result execute(CqnInsert insert);

  /**
   * Executes a {@link CqnUpsert} statement.
   *
   * @param upsert the CQN upsert statement to be executed
   * @return the {@link Result} of the upsert
   */
  Result execute(CqnUpsert upsert);

  /**
   * Executes a {@link CqnUpdate} statement.
   *
   * @param update the CQN update statement to be executed
   * @return the {@link Result} of the update
   */
  default Result execute(CqnUpdate update) {
    return execute(update, emptyMap());
  }

  /**
   * Executes a {@link CqnUpdate} statement with optional positional values for indexed parameters,
   * see {@link CQL#param(int)}.
   *
   * @param update the CQN update statement to be executed
   * @param paramValues the positional parameter values
   * @return the {@link Result} of the update
   */
  Result execute(CqnUpdate update, Object... paramValues);

  /**
   * Executes a {@link CqnUpdate} statement with values for named parameters, see {@link
   * CQL#param(String)}.
   *
   * @param update the CQN update statement to be executed
   * @param namedValues the named parameter values
   * @return the {@link Result} of the update
   */
  Result execute(CqnUpdate update, Map<String, Object> namedValues);

  /**
   * Executes a {@link CqnUpdate} statement as batch with the given named parameter values.
   *
   * @param update the CQN update statement to be executed
   * @param valueSets the named parameter value sets
   * @return the {@link Result} of the batch update
   */
  Result execute(CqnUpdate update, Iterable<Map<String, Object>> valueSets);

  /**
   * Executes a {@link CqnDelete} statement .
   *
   * @param delete the CQN delete statement to be executed
   * @return the {@link Result} of the delete
   */
  default Result execute(CqnDelete delete) {
    return execute(delete, emptyMap());
  }

  /**
   * Executes a {@link CqnDelete} statement with optional positional values for indexed parameters,
   * see {@link CQL#param(int)}.
   *
   * @param delete the CQN delete statement to be executed
   * @param paramValues the positional parameter values
   * @return the {@link Result} of the delete
   */
  Result execute(CqnDelete delete, Object... paramValues);

  /**
   * Executes a {@link CqnDelete} statement with values for named parameters, see {@link
   * CQL#param(String)}.
   *
   * @param delete the CQN delete statement to be executed
   * @param namedValues the named parameter values
   * @return the {@link Result} of the delete
   */
  Result execute(CqnDelete delete, Map<String, Object> namedValues);

  /**
   * Executes a {@link CqnDelete} statement as batch with the given named parameter values.
   *
   * @param delete the CQN delete statement to be executed
   * @param valueSets the named parameter value sets
   * @return the {@link Result} of the batch delete
   */
  Result execute(CqnDelete delete, Iterable<Map<String, Object>> valueSets);

  SessionContext getSessionContext();

  void setSessionContext(SessionContext session);

  @Beta
  default void deleteAll(Stream<CdsEntity> entities) {
    entities.map(Delete::from).forEach(this::execute);
  }
}
