/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds;

import com.sap.cds.ql.CdsDataException;
import com.sap.cds.reflect.CdsEntity;
import java.io.Serial;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Thrown by the data store, if a unique constraint is violated when executing an insert, upsert or
 * update statement.
 */
public class UniqueConstraintException extends CdsDataException {
  @Serial private static final long serialVersionUID = 1L;

  private final transient CdsEntity entity;
  private final transient Map<String, Object> elementValues;

  public UniqueConstraintException(
      CdsEntity entity, Map<String, Object> elementValues, Set<String> keys, Exception ex) {
    super(
        "Unique constraint violated when inserting or updating '%s[%s]'"
            .formatted(entity.getQualifiedName(), stringify(elementValues, keys)),
        ex);
    this.entity = entity;
    this.elementValues = elementValues;
  }

  /**
   * Get entity qualified name
   *
   * @return entity qualified name
   */
  public String getEntityName() {
    return entity.getQualifiedName();
  }

  /**
   * Get map of element names and values violating the unique constraint
   *
   * @return map of element names and values
   */
  public Map<String, Object> getElementValues() {
    return elementValues;
  }

  private static String stringify(Map<String, Object> values, Set<String> keys) {
    return values.entrySet().stream()
        .map(k -> k.getKey() + (keys.contains(k.getKey()) ? "=" + k.getValue() : ""))
        .collect(Collectors.joining(", "));
  }
}
