/*
 * © 2018-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql.cqn;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.cqn.transformation.CqnTransformation;
import java.util.List;
import java.util.Optional;

public interface CqnSelect extends CqnFilterableStatement, CqnEntitySelector, CqnSource {

  /**
   * The from clause of this select statement.
   *
   * @return a reference to this statement's {@link CqnSource}
   */
  CqnSource from();

  /**
   * @return whether statement is a SELECT DISTINCT
   */
  boolean isDistinct();

  /**
   * @deprecated instead use {@link CqnEntitySelector#items}
   * @return the select list items
   */
  @Deprecated
  default List<CqnSelectListItem> columns() {
    return items();
  }

  List<CqnValue> groupBy();

  List<String> excluding();

  Optional<CqnPredicate> having();

  Optional<CqnLock> getLock();

  Optional<CqnPredicate> search();

  /**
   * Returns the pipeline of transformations, which is applied to the source ref of this select
   * statement before the the regular clauses are applied.
   *
   * <p>The statement is processed in this order:
   *
   * <ol>
   *   <li>ref, source
   *   <li>transformations
   *   <li>where, search
   *   <li>groupBy
   *   <li>having
   *   <li>items, distinct
   *   <li>orderBy
   *   <li>skip
   *   <li>top
   * </ol>
   *
   * @return the pipeline of transformations
   */
  @Beta
  List<CqnTransformation> transformations();

  @Override
  default boolean isSelect() {
    return true;
  }

  @Override
  default CqnSelect asSelect() {
    return this;
  }

  /**
   * Traverses the clauses of this {@code CqnSelect} with a given {@code visitor}.
   *
   * <p>If the source of this select is a structured type reference it is visited first.
   *
   * <p>The other clauses are traversed depth-first in the following order:
   *
   * <ul>
   *   <li>items
   *   <li>where
   *   <li>search
   *   <li>group by
   *   <li>having
   *   <li>order by
   * </ul>
   *
   * Afterwards this {@code CqnSelect} is passed to the {@link CqnVisitor#visit(CqnSelect)} method.
   *
   * @param visitor the {@link CqnVisitor}
   */
  @Override
  default void accept(CqnVisitor visitor) {
    if (from().isRef()) {
      ref().accept(visitor);
    }

    dispatch(visitor);

    visitor.visit(this);
  }

  @Override
  default void dispatch(CqnVisitor visitor) {
    items().forEach(c -> c.accept(visitor));
    where().ifPresent(w -> w.accept(visitor));
    search().ifPresent(s -> s.accept(visitor));
    groupBy().forEach(c -> c.accept(visitor));
    having().ifPresent(h -> h.accept(visitor));
    orderBy().forEach(o -> o.accept(visitor));
  }
}
