/*
 * © 2018-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql.cqn;

import com.google.common.annotations.Beta;

/**
 * A {@link CqnToken} tree visitor conform to the visitor design pattern. Classes implementing this
 * interface operate on a tree of {@code CqnTokens}, such as CQN expressions, predicates and values.
 *
 * <p>When a visitor is passed as an argument to a token's {@code accept} method, generally the
 * {@code accept} methods of the token's children are called first. Afterwards the {@code visit}
 * method that is most specific to the parent token is invoked. Classes implementing this interface
 * may override the default {@code visit} method to perform arbitrary operations.
 *
 * <p>Consider the following example with the visitor implementation:
 *
 * <pre>
 * CqnVisitor visitor = new CqnVisitor() {
 * 	&#64;Override
 * 	public void visit(CqnComparisonPredicate comparison) {
 * 		System.out.println(comparison.operator());
 * 	}
 *
 * 	&#64;Override
 * 	public void visit(CqnElementRef elementRef) {
 * 		System.out.println(elementRef.displayName());
 * 	}
 *
 * 	&#64;Override
 * 	public void visit(CqnStringLiteral literal) {
 * 		System.out.println(literal.value());
 * 	}
 * };
 * </pre>
 *
 * and the comparison predicate:
 *
 * <pre>
 * Predicate compPredicate = CQL.comparison(CQL.get("name"), Operator.IS, CQL.constant("Peter"));
 * </pre>
 *
 * Calling the {@code compPredicate.accept(visitor)} will produce the following output:
 *
 * <pre>
 * name
 * Peter
 * IS
 * </pre>
 *
 * Note the order in which the nodes are visited. As per {@link CqnComparisonPredicate#accept} the
 * visitor is first dispatched to the left and right values of the comparison. Afterwards the {@code
 * visit(CqnComparisonPredicate comparison)} method is called.
 */
public interface CqnVisitor {

  /**
   * Called for each traversed {@link CqnFunc function} call
   *
   * @param func the function call
   */
  default void visit(CqnFunc func) {}

  /**
   * Called for each traversed {@link CqnWindowFunc window function} call
   *
   * @param func the window function call
   */
  default void visit(CqnWindowFunc func) {
    visit((CqnFunc) func);
  }

  /**
   * Called for each traversed {@link CqnStructuredTypeRef structured type reference}
   *
   * @param typeRef the structured type reference
   */
  default void visit(CqnStructuredTypeRef typeRef) {}

  /**
   * Called for each traversed {@link CqnElementRef element reference}
   *
   * @param elementRef the element reference
   */
  default void visit(CqnElementRef elementRef) {}

  /**
   * Called for each traversed {@link CqnPlain plain value}
   *
   * @param plain the plain value
   */
  default void visit(CqnPlain plain) {}

  /**
   * Called for each traversed {@link CqnStar star token}
   *
   * @param star the star token
   */
  default void visit(CqnStar star) {
    visit((CqnSelectListItem) star);
  }

  /**
   * Called for each traversed {@link CqnSelectListValue select list value}
   *
   * @param slv the select list value
   */
  default void visit(CqnSelectListValue slv) {
    visit((CqnSelectListItem) slv);
  }

  /**
   * Called for each traversed {@link CqnSelectListItem select list item} unless the more specific
   * methods {@link #visit(CqnSelectListValue)} or {@link #visit(CqnStar)} are implemented
   *
   * @param sli the select list item
   */
  default void visit(CqnSelectListItem sli) {}

  /**
   * Called for each traversed {@link CqnSortSpecification sort specification}
   *
   * @param sortSpec the sort specification
   */
  default void visit(CqnSortSpecification sortSpec) {}

  /**
   * Called for each traversed {@link CqnExpression expression} unless the more specific methods
   * {@link #visit(CqnArithmeticExpression)}, {@link #visit(CqnArithmeticNegation)} or {@link
   * #visit(CqnPredicate)} are implemented
   *
   * @param expr the expression
   */
  default void visit(CqnExpression expr) {}

  /**
   * Called for each traversed {@link CqnCaseExpression case expression}
   *
   * @param expr the arithmetic expression
   */
  default void visit(CqnCaseExpression expr) {
    visit((CqnExpression) expr);
  }

  /**
   * Called for each traversed {@link CqnArithmeticExpression arithmetic expression}
   *
   * @param expr the arithmetic expression
   */
  default void visit(CqnArithmeticExpression expr) {
    visit((CqnExpression) expr);
  }

  /**
   * Called for each traversed {@link CqnArithmeticNegation arithmetic negation}
   *
   * @param neg the arithmetic negation
   */
  default void visit(CqnArithmeticNegation neg) {
    visit((CqnExpression) neg);
  }

  /**
   * Called for each traversed {@link CqnPredicate predicate} unless the more specific methods
   * {@link #visit(CqnComparisonPredicate)}, {@link #visit(CqnSearchTermPredicate)}, {@link
   * #visit(CqnPassThroughSearchPredicate)}, {@link #visit(CqnConnectivePredicate)}, {@link
   * #visit(CqnInPredicate)}, {@link #visit(CqnNegation)}, {@link #visit(CqnExistsSubquery)}, {@link
   * #visit(CqnMatchPredicate)} or {@link #visit(CqnMatchPredicate)} are implemented
   *
   * @param pred the predicate
   */
  default void visit(CqnPredicate pred) {
    visit((CqnExpression) pred);
  }

  /**
   * Called for each traversed {@link CqnComparisonPredicate comparison predicate}
   *
   * @param comparison the comparison predicate
   */
  default void visit(CqnComparisonPredicate comparison) {
    visit((CqnPredicate) comparison);
  }

  /**
   * Called for each traversed {@link CqnSearchTermPredicate search predicate}
   *
   * @param search the search predicate
   */
  @SuppressWarnings("removal")
  default void visit(CqnSearchTermPredicate search) {
    search.tokens().forEach(token -> token.accept(this));
    visit((CqnSearchPredicate) search);
  }

  /**
   * @deprecated use {@link #visit(CqnSearchTermPredicate)} instead
   */
  @Deprecated(since = "3.0", forRemoval = true)
  default void visit(CqnSearchPredicate search) {
    visit((CqnPredicate) search);
  }

  /**
   * Called for each traversed {@link CqnPassThroughSearchPredicate search predicate}
   *
   * @param search the pass-through search predicate
   */
  default void visit(CqnPassThroughSearchPredicate search) {
    visit((CqnPredicate) search);
  }

  /**
   * Called for each traversed {@link CqnConnectivePredicate connective predicate}, i.e. logical
   * conjunction or disjunction
   *
   * @param connective the connective predicate
   */
  default void visit(CqnConnectivePredicate connective) {
    visit((CqnPredicate) connective);
  }

  /**
   * Called for each traversed {@link CqnInPredicate IN predicate}
   *
   * @param in the IN predicate
   */
  default void visit(CqnInPredicate in) {
    visit((CqnPredicate) in);
  }

  /**
   * Called for each traversed {@link CqnInSubquery IN subquery predicate}
   *
   * @param in the IN subquery predicate
   */
  @Beta
  default void visit(CqnInSubquery in) {
    visit((CqnPredicate) in);
  }

  /**
   * Called for each traversed {@link CqnBetweenPredicate BETWEEN predicate}
   *
   * @param between the BETWEEN predicate
   */
  default void visit(CqnBetweenPredicate between) {
    visit((CqnPredicate) between);
  }

  /**
   * Called for each traversed {@link CqnEtagPredicate ETag predicate}
   *
   * @param etag the ETag predicate
   */
  @Beta
  default void visit(CqnEtagPredicate etag) {
    visit((CqnInPredicate) etag);
  }

  /**
   * Called for each traversed {@link CqnNegation logical negation}
   *
   * @param neg the logical negation
   */
  default void visit(CqnNegation neg) {
    visit((CqnPredicate) neg);
  }

  /**
   * Called for each traversed {@link CqnContainmentTest containment test}
   *
   * @param test the containment test
   */
  default void visit(CqnContainmentTest test) {
    visit((CqnFunc) test);
  }

  /**
   * Called for each traversed {@link CqnLiteral literal} unless the more specific methods {@link
   * #visit(CqnBooleanLiteral)}, {@link #visit(CqnNumericLiteral)}, {@link #visit(CqnStringLiteral)}
   * or {@link #visit(CqnTemporalLiteral)} are implemented
   *
   * @param literal the literal
   */
  default void visit(CqnLiteral<?> literal) {}

  /**
   * Called for each traversed {@link CqnStringLiteral string literal}
   *
   * @param string the string literal
   */
  default void visit(CqnStringLiteral string) {
    visit((CqnLiteral<?>) string);
  }

  /**
   * Called for each traversed {@link CqnBooleanLiteral Boolean literal}
   *
   * @param bool the Boolean literal
   */
  default void visit(CqnBooleanLiteral bool) {
    visit((CqnLiteral<?>) bool);
  }

  /**
   * Called for each traversed {@link CqnNumericLiteral numeric literal}
   *
   * @param number the numeric literal
   */
  default void visit(CqnNumericLiteral<?> number) {
    visit((CqnLiteral<?>) number);
  }

  /**
   * Called for each traversed {@link CqnTemporalLiteral temporal literal}
   *
   * @param temporal the temporal literal
   */
  default void visit(CqnTemporalLiteral<?> temporal) {
    visit((CqnLiteral<?>) temporal);
  }

  /**
   * Called for each traversed {@link CqnVector vector}
   *
   * @param vector the vector
   */
  default void visit(CqnVector vector) {
    visit((CqnLiteral<?>) vector);
  }

  /**
   * Called for each traversed {@link CqnNullValue NULL value}
   *
   * @param nil the NULL value
   */
  default void visit(CqnNullValue nil) {}

  /**
   * Called for each traversed {@link CqnParameter parameter}
   *
   * @param param the parameter
   */
  default void visit(CqnParameter param) {}

  /**
   * Called for each traversed {@link CqnInline inline specification}
   *
   * @param inline the inline specification
   */
  default void visit(CqnInline inline) {}

  /**
   * Called for each traversed {@link CqnExpand expand specification}
   *
   * @param expand the expand specification
   */
  default void visit(CqnExpand expand) {
    visit((CqnEntitySelector) expand);
  }

  /**
   * Called for each traversed {@link CqnExistsSubquery exists subquery}
   *
   * @param exists the exists subquery
   */
  default void visit(CqnExistsSubquery exists) {
    visit((CqnPredicate) exists);
  }

  /**
   * Called for each traversed {@link CqnMatchPredicate match predicate}
   *
   * @param match the match predicate
   */
  default void visit(CqnMatchPredicate match) {
    visit((CqnPredicate) match);
  }

  /**
   * Called for each traversed {@link CqnListValue list value}
   *
   * @param list the list value
   */
  default void visit(CqnListValue list) {}

  default void visit(CqnSelect select) {
    visit((CqnEntitySelector) select);
  }

  /**
   * Called for each traversed {@link CqnEntitySelector entity selector} unless the more specific
   * methods {@link #visit(CqnExpand)} or {@link #visit(CqnSelect)} are implemented
   *
   * @param selector the entity selector
   */
  default void visit(CqnEntitySelector selector) {}
}
