/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql.hana;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.Source;
import com.sap.cds.ql.TableFunction;
import com.sap.cds.ql.cqn.CqnPredicate;

/**
 * A function to compute the subset of a source hierarchy, which contains all nodes relative
 * (ancestors or descendants) to a set of start nodes.
 *
 * <p>The function adds the element {@code hierarchy_distance : Int32} to the result, which
 * indicates the distance of the retained node from the start node.
 *
 * <p>Nodes can occur multiple times in the result if they are reachable from multiple start nodes.
 */
@Beta
public interface HierarchySubset extends CqnHierarchySubset, TableFunction {

  /**
   * Creates a copy of this hierarchy subset function with the given source.
   *
   * @param source the hierarchy source
   * @return new hierarchy subset function with given source
   */
  HierarchySubset copy(Source<?> source);

  /**
   * Specifies a filter condition that identifies the start nodes of this hierarchy subset function.
   *
   * @param startWhere the filter condition that specifies the start nodes
   * @return this hierarchy subset function
   */
  HierarchySubset startWhere(CqnPredicate startWhere);

  /**
   * Specifies a minimum and a maximum HIERARCHY_DISTANCE
   *
   * @param from the minimum distance
   * @param to the maximum distance
   * @return this hierarchy subset function
   */
  HierarchySubset distance(int from, int to);

  /**
   * Specifies a minimum HIERARCHY_DISTANCE
   *
   * @param from the minimum distance
   * @return this hierarchy subset function
   */
  HierarchySubset from(int from);

  /**
   * Specifies a maximum HIERARCHY_DISTANCE
   *
   * @param to the maximum distance
   * @return this hierarchy subset function
   */
  HierarchySubset to(int to);

  /**
   * Specifies a maximum absolute HIERARCHY_DISTANCE
   *
   * @param distanceFromStart the maximum absolute HIERARCHY_DISTANCE
   * @param keepStart whether the start nodes should be included
   * @return this hierarchy subset function
   */
  HierarchySubset distance(int distanceFromStart, boolean keepStart);
}
