/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.reflect;

import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toMap;

import com.google.common.annotations.Beta;
import com.sap.cds.CdsData;
import com.sap.cds.CdsException;
import com.sap.cds.CdsVector;
import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.Map;

public enum CdsBaseType {
  UUID("cds.UUID", String.class),
  BOOLEAN("cds.Boolean", Boolean.class),
  INTEGER("cds.Integer", Integer.class),
  UINT8("cds.UInt8", Short.class),
  INT16("cds.Int16", Short.class),
  INT32("cds.Int32", Integer.class),
  INT64("cds.Int64", Long.class),
  INTEGER64("cds.Integer64", Long.class),
  DECIMAL("cds.Decimal", BigDecimal.class), //
  @Deprecated
  // use DECIMAL without length
  DECIMAL_FLOAT("cds.DecimalFloat", BigDecimal.class), //
  DOUBLE("cds.Double", Double.class),
  DATE("cds.Date", LocalDate.class),
  TIME("cds.Time", LocalTime.class),
  DATETIME("cds.DateTime", Instant.class),
  TIMESTAMP("cds.Timestamp", Instant.class),
  STRING("cds.String", String.class),
  BINARY("cds.Binary", byte[].class),
  LARGE_STRING("cds.LargeString", String.class),
  LARGE_BINARY("cds.LargeBinary", byte[].class),
  @Beta
  VECTOR("cds.Vector", CdsVector.class),
  @Beta
  MAP("cds.Map", CdsData.class),
  HANA_SMALLINT("cds.hana.SMALLINT", Short.class),
  HANA_TINYINT("cds.hana.TINYINT", Short.class),
  HANA_SMALLDECIMAL("cds.hana.SMALLDECIMAL", BigDecimal.class),
  HANA_REAL("cds.hana.REAL", Float.class),
  HANA_CHAR("cds.hana.CHAR", String.class),
  HANA_NCHAR("cds.hana.NCHAR", String.class),
  HANA_VARCHAR("cds.hana.VARCHAR", String.class),
  HANA_CLOB("cds.hana.CLOB", String.class),
  HANA_BINARY("cds.hana.BINARY", byte[].class), //
  @Beta
  // this mapping might change in the future
  HANA_ST_POINT("cds.hana.ST_POINT", byte[].class), //
  @Beta
  // this mapping might change in the future
  HANA_ST_GEOMETRY("cds.hana.ST_GEOMETRY", byte[].class),
  @Beta
  UNDEFINED("undefined", Object.class);

  private static final Map<String, CdsBaseType> reverse =
      stream(values()).collect(toMap(CdsBaseType::cdsName, t -> t));

  private final String cdsName;
  private final Class<?> javaType;

  private CdsBaseType(String cdsName, Class<?> javaType) {
    this.cdsName = cdsName;
    this.javaType = javaType;
  }

  public String cdsName() {
    return cdsName;
  }

  public Class<?> javaType() {
    return javaType;
  }

  @Override
  public String toString() {
    return cdsName;
  }

  public static CdsBaseType cdsType(String cdsName) {
    return reverse.getOrDefault(cdsName, UNDEFINED);
  }

  public static Class<?> cdsJavaMediaType(CdsBaseType cdsType) {
    switch (cdsType) {
      case HANA_CLOB:
      case STRING:
      case LARGE_STRING:
        return Reader.class;
      case LARGE_BINARY:
        return InputStream.class;
      default:
        throw new CdsException("@Core.MediaType is not supported for type " + cdsType);
    }
  }
}
