/*
 * © 2019-2023 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.reflect;

import java.util.Optional;
import java.util.stream.Stream;

public interface CdsStructuredType extends CdsType {

  /**
   * Get a {@link CdsElement} by name.
   *
   * @param name the name of the {@code CdsElement}
   * @return the {@code CdsElement} with the given name
   * @throws CdsElementNotFoundException if there is no {@code CdsElement} with the given name
   */
  CdsElement getElement(String name);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsElement}.
   *
   * @param name the name of the {@code CdsElement}
   * @return an {@code Optional} describing the {@code CdsElement} with the given name, or an empty
   *     {@code Optional} if there is no {@code CdsElement} with this name
   */
  Optional<CdsElement> findElement(String name);

  /**
   * Get a {@link CdsElement} of type {@link CdsAssociationType} by name.
   *
   * @param name the name of the {@code CdsElement}
   * @return the {@code CdsElement} of type {@code CdsAssociation} with the given name
   * @throws CdsElementNotFoundException if there is no {@code CdsElement} with the given name
   */
  CdsElement getAssociation(String name);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsElement} of type {@link CdsAssociationType}.
   *
   * @param name the name of the {@code CdsElement}
   * @return an {@code Optional} describing the {@code CdsElement} of type {@code CdsAssociation}
   *     with the given name, or an empty {@code Optional} if there is no {@code CdsElement} with
   *     this name
   */
  Optional<CdsElement> findAssociation(String name);

  /**
   * Returns the target {@link CdsStructuredType} of the given navigation path.
   *
   * @param path the navigation path
   * @return the target {@code CdsStructuredType} of the path
   * @throws CdsReflectiveOperationException if the path is invalid
   */
  <S extends CdsStructuredType> S getTargetOf(String path);

  /**
   * Returns a sequential {@code Stream} over all {@link CdsElement CdsElement(s)} in this
   * structured type.
   *
   * @return a sequential {@code Stream} over the {@code CdsElement(s)}
   */
  Stream<CdsElement> elements();

  /**
   * Returns a sequential {@code Stream} over the key {@link CdsElement CdsElement(s)} in this
   * structured type.
   *
   * @return a sequential {@code Stream} over the key {@code CdsElement(s)}
   */
  default Stream<CdsElement> keyElements() {
    return elements().filter(CdsElement::isKey);
  }

  /**
   * Returns a sequential {@code Stream} over the virtual {@link CdsElement CdsElement(s)} in this
   * structured type.
   *
   * @return a sequential {@code Stream} over the virtual {@code CdsElement(s)}
   */
  default Stream<CdsElement> virtualElements() {
    return elements().filter(CdsElement::isVirtual);
  }

  /**
   * Returns a sequential {@code Stream} over the non virtual {@link CdsElement CdsElement(s)} in
   * this structured type.
   *
   * @return a sequential {@code Stream} over the non virtual {@code CdsElement(s)}
   */
  default Stream<CdsElement> concreteElements() {
    return elements().filter(e -> !e.isVirtual());
  }

  /**
   * Returns a sequential {@code Stream} over the {@link CdsElement CdsElement(s)} in this
   * structured type that are not of type {@link CdsAssociationType}.
   *
   * @return a sequential {@code Stream} over the {@code CdsElement(s)} that are not of type {@code
   *     CdsAssociation}
   */
  default Stream<CdsElement> nonAssociationElements() {
    return elements().filter(e -> !e.getType().isAssociation());
  }

  /**
   * Returns a sequential {@code Stream} over the {@link CdsElement CdsElement(s)} in this
   * structured type that are not of type {@link CdsAssociationType} and not virtual.
   *
   * @return a sequential {@code Stream} over the {@code CdsElement(s)} that are not of type {@code
   *     CdsAssociation}
   */
  default Stream<CdsElement> concreteNonAssociationElements() {
    return elements().filter(e -> !e.getType().isAssociation() && !e.isVirtual());
  }

  /**
   * Returns a sequential {@code Stream} over the {@link CdsElement CdsElement(s)} of type {@link
   * CdsAssociationType} in this structured type.
   *
   * @return a sequential {@code Stream} over the {@code CdsElement(s)} of type {@code
   *     CdsAssociation}
   */
  default Stream<CdsElement> associations() {
    return elementsOfType(CdsAssociationType.class);
  }

  /**
   * Returns a sequential {@code Stream} over the {@link CdsElement CdsElement(s)} of type
   * composition in this structured type.
   *
   * @return a sequential {@code Stream} over the {@code CdsElement(s)} of type composition
   */
  default Stream<CdsElement> compositions() {
    return associations().filter(a -> a.getType().as(CdsAssociationType.class).isComposition());
  }

  /**
   * Returns a sequential {@code Stream} over the {@link CdsElement CdsElement(s)} of the given
   * {@code type} in this structured type.
   *
   * @param <T> the type of the {@code CdsElement(s)}
   * @param type the type class of the {@code CdsElement(s)}
   * @return a sequential {@code Stream} over the {@code CdsElement(s)} of the given type {@code
   *     <T>}
   */
  default <T extends CdsType> Stream<CdsElement> elementsOfType(Class<T> type) {
    return elements().filter(e -> type.isAssignableFrom(e.getType().getClass()));
  }

  /**
   * Returns {@code true} if this is an inline defined structured type without a name.
   *
   * @return {@code true} if this is a structured type without name, otherwise {@code false}
   */
  default boolean isAnonymous() {
    return getQualifiedName() == null || getQualifiedName().isEmpty();
  }

  @Override
  default boolean isStructured() {
    return true;
  }

  @Override
  default void accept(CdsVisitor visitor) {
    elements().forEach(e -> e.accept(visitor));

    visitor.visit(this);
  }
}
