/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.reflect.overlay;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.reflect.CdsElementNotFoundException;
import com.sap.cds.reflect.CdsEntity;

/**
 * The {@code CdsEntityExtender} allows to extend the underlying {@link CdsEntity} with additional
 * elements, associations/compositions and annotations or extend its existing elements.
 */
@Beta
public interface CdsEntityExtender {

  /**
   * Creates a new {@link CdsElementExtender} for an element with the given name to be added to the
   * entity. If the entity has an element with the same name already, an {@link
   * IllegalArgumentException} is thrown.
   *
   * @param name the name of the element to add
   * @return a {@code CdsElementExtender} to further configure the element
   * @throws IllegalArgumentException if an element with the same name already exists
   */
  CdsElementExtender addElement(String name);

  /**
   * Returns a new {@link CdsElementExtender} for an element with the given name. If the element
   * does not exist in the entity, a {@link CdsElementNotFoundException} is thrown.
   *
   * @param name the name of the element to extend
   * @return a {@code CdsElementExtender} to further configure the element
   * @throws CdsElementNotFoundException if the element does not exist in the entity
   */
  CdsElementExtender element(String name);

  /**
   * Returns a new {@link CdsAssociationExtender} for an association with the given name to be added
   * to the entity. If the entity has an element with the same name already, an {@link
   * IllegalArgumentException} is thrown.
   *
   * @param name the name of the association to add
   * @return a {@code CdsAssociationExtender} to further configure the association
   * @throws IllegalArgumentException if an element with the same name already exists
   */
  CdsAssociationExtender addAssociation(String name);

  /**
   * Returns a new {@link CdsAssociationExtender} for a composition with the given name to be added
   * to the entity. If the entity has an element with the same name already, an {@link
   * IllegalArgumentException} is thrown.
   *
   * @param name the name of the composition to add
   * @return a {@code CdsAssociationExtender} to further configure the composition
   * @throws IllegalArgumentException if an element with the same name already exists
   */
  CdsAssociationExtender addComposition(String name);

  /**
   * Adds an annotation to the entity.
   *
   * @param key the annotation key
   * @param value the annotation value
   * @return this {@code CdsEntityExtender}
   */
  CdsEntityExtender addAnnotation(String key, Object value);

  /**
   * Sets the {@link CqnSelect} query for the entity.
   *
   * @param query the query
   * @return this {@code CdsEntityExtender}
   */
  CdsEntityExtender query(CqnSelect query);

  /**
   * Returns to the {@link CdsModelExtender} that created this {@code CdsEntityExtender}.
   *
   * @return the {@code CdsModelExtender} to continue extending the model
   */
  CdsModelExtender and();
}
