/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.ql;

import static java.util.Arrays.asList;

import java.util.List;
import java.util.function.Function;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnVisitor;

public interface Expand<T extends StructuredType<?>> extends CqnExpand {

	@Override
	StructuredTypeRef ref();

	/**
	 * Sets the limit of the expand.
	 * 
	 * @param top the maximum number of entries returned by the expand or -1 if
	 *            unlimited
	 * @return this expand
	 */
	default Expand<T> limit(long top) {
		return limit(top, 0);
	}

	/**
	 * Sets the limit and offset of the expand.
	 * 
	 * @param top  the maximum number of entries returned by the expand or -1 if
	 *             unlimited
	 * @param skip the number of entries that shall be skipped
	 * @return this expand
	 */
	Expand<T> limit(long top, long skip);

	Expand<T> items(Iterable<? extends Selectable> items);

	Expand<T> orderBy(String... refs);

	Expand<T> orderBy(List<CqnSortSpecification> sortSpec);

	Expand<T> orderBy(Function<T, CqnSortSpecification> sort);

	Expand<T> orderBy(Function<T, CqnSortSpecification> sort1, Function<T, CqnSortSpecification> sort2);

	Expand<T> orderBy(Function<T, CqnSortSpecification> sort1, Function<T, CqnSortSpecification> sort2,
			Function<T, CqnSortSpecification> sort3);

	Expand<T> orderBy(Function<T, CqnSortSpecification> sort1, Function<T, CqnSortSpecification> sort2,
			Function<T, CqnSortSpecification> sort3, Function<T, CqnSortSpecification> sort4);

	Expand<T> orderBy(Function<T, CqnSortSpecification> sort1, Function<T, CqnSortSpecification> sort2,
			Function<T, CqnSortSpecification> sort3, Function<T, CqnSortSpecification> sort4,
			Function<T, CqnSortSpecification> sort5);

	@SuppressWarnings("unchecked")
	Expand<T> orderBy(Function<T, CqnSortSpecification>... sortSpec);

	/**
	 * Specifies that the total number of entities matching the expand ref shall be
	 * returned along with the expand result.
	 *
	 * Upon the calculation of the inline count the limit clause of the expand is
	 * ignored.
	 *
	 * Inline count is not supported in combination with distinct and groupBy.
	 * 
	 * The inline count is currently added to the result under the alias of the
	 * expand suffixed with "@count". This may be subject to change in future
	 * releases.
	 *
	 * @return this expand
	 */
	@Beta
	Expand<T> inlineCount();

	/**
	 * Sets the inline count flag of this expand.
	 * 
	 * @param inlineCount true to enable inline count, false to disable
	 * @return this expand
	 * 
	 * @see #inlineCount()
	 */
	@Beta
	Expand<T> inlineCount(boolean inlineCount);

	default Expand<T> orderBy(CqnSortSpecification... sortSpec) {
		return orderBy(asList(sortSpec));
	}

	/**
	 * Sets the alias of the expand.
	 * 
	 * @param alias the alias, or null
	 * @return this expand
	 */
	Expand<T> as(String alias);

	@Override
	default void accept(CqnVisitor visitor) {
		CqnExpand.super.accept(visitor);
	}

}
