/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.ql;

import java.util.Collection;
import java.util.Map;
import java.util.function.Function;
import java.util.function.UnaryOperator;

import com.sap.cds.CdsDataStore;
import com.sap.cds.ql.cqn.CqnExpression;
import com.sap.cds.ql.cqn.CqnParameter;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;

/**
 * The Update Builder allows to construct CDS QL update statements, which can be
 * executed via the {@link CdsDataStore}.
 *
 * Note that some methods in this interface mutually exclude each other and
 * cannot be combined. For example, the methods {@link #byId(Object) byId},
 * {@link #byParams(String...) byParams}, {@link #matching(Map) matching}, and
 * {@link #where(CqnPredicate) where} all set the where condition of the
 * enclosing statement, overwriting any previously stored conditions.
 *
 * @param <T> the type of the entity set targeted by this update statement
 */
public interface Update<T extends StructuredType<?>> extends CqnUpdate, FilterableStatement<T, Update<T>> {

	/**
	 * Creates an update statement to update entries of a specified entity set.
	 *
	 * @param qualifiedName the fully qualified name of the CDS entity set
	 * @return the update statement
	 */
	static Update<StructuredType<?>> entity(String qualifiedName) {
		return entity(qualifiedName, e -> e);
	}

	/**
	 * Creates an update statement to update entries of a specified entity set.
	 *
	 * @param ref the ref to the entity
	 * @return the update statement
	 */
	static Update<StructuredType<?>> entity(CqnStructuredTypeRef ref) {
		return CDS.QL.builder.update(ref);
	}

	/**
	 * Creates an update statement to update entries from a specified entity set.
	 *
	 * @param <T>    the type of the entity set
	 * @param entity the structured type representing the entity set
	 * @return the update statement
	 */
	public static <T extends StructuredType<?>> Update<T> entity(T entity) {
		return CDS.QL.builder.update(entity);
	}

	/**
	 * Creates an update statement to update entries of a specified entity set.
	 *
	 * @param qualifiedName the fully qualified name of the CDS entity set
	 * @param path          a path expression navigating from the root entity to the
	 *                      target entity of the update statement
	 * @return the update statement
	 */
	static Update<StructuredType<?>> entity(String qualifiedName, UnaryOperator<StructuredType<?>> path) {
		return CDS.QL.builder.update(qualifiedName, path);
	}

	/**
	 * Creates an update statement to update data for a specified entity set.
	 *
	 * @param <T>    the type of the entity set
	 * @param entity the static model representation of the entity set
	 * @return the update statement
	 */
	static <T extends StructuredType<T>> Update<T> entity(Class<T> entity) {
		return entity(entity, e -> e);
	}

	/**
	 * Creates an update statement to update data for a specified entity set.
	 *
	 * @param <R>    the type of the root entity
	 * @param <T>    the type of the target entity
	 * @param entity the static model representation of the entity set
	 * @param path   a path expression navigating from the root entity to the target
	 *               entity of the update statement
	 * @return the update statement
	 */
	static <R extends StructuredType<R>, T extends StructuredType<T>> Update<T> entity(Class<R> entity,
			Function<R, T> path) {
		return CDS.QL.builder.update(entity, path);
	}

	/**
	 * Creates an update statement to update entries of a specified entity set.
	 *
	 * @param entity the model representation of the entity set obtained by
	 *               reflection
	 * @param path   a path expression navigating from the root entity to the target
	 *               entity of the update statement
	 * @return the update statement
	 * @see com.sap.cds.reflect.CdsModel#findEntity(String)
	 * @see com.sap.cds.reflect.CdsModel#entities()
	 */
	static Update<StructuredType<?>> entity(CdsEntity entity, UnaryOperator<StructuredType<?>> path) {
		return CDS.QL.builder.update(entity, path);
	}

	/**
	 * Creates an update statement to update data for a specified entity set.
	 *
	 * @param entity the model representation of the entity set obtained by
	 *               reflection
	 * @return the update statement
	 * @see com.sap.cds.reflect.CdsModel#findEntity(String)
	 * @see com.sap.cds.reflect.CdsModel#entities()
	 */
	static Update<StructuredType<?>> entity(CdsEntity entity) {
		return entity(entity, e -> e);
	}

	/**
	 * Copies the given {@link CqnUpdate} into an {@link Update} builder.
	 *
	 * @param update the {@code CqnUpdate} to be copied
	 * @return the modifiable update statement copy
	 */
	static Update<StructuredType<?>> copy(CqnUpdate update) {
		return CDS.QL.builder.copy(update);
	}

	/**
	 * Creates an update statement to update data for a specified entity set.
	 *
	 * @param model the CDS model
	 * @param cqnUpdate the CQN representation of the update statement
	 * @return the update statement
	 */
	static Update<StructuredType<?>> cqn(CdsModel model, String cqnUpdate) {
		return CDS.QL.parser.update(model, cqnUpdate);
	}

	/**
	 * Sets the batch data to be updated by the update statement. The data is given
	 * as a collection of maps from element names of the target entity set to their
	 * new values. The data must contain the entities' keys.
	 *
	 * @param entries a collection of data to be updated given as a map from element
	 *                name to new value
	 * @return the update statement
	 */
	Update<T> entries(Iterable<? extends Map<String, ?>> entries);

	/**
	 * Sets the entry to be updated by the update statement. The entry is given as a
	 * map of element names of the target entity set to their new values. The entry
	 * must contain key value mappings to identify the entity to be updated.
	 *
	 * @param entry the data to be updated as a map from element name to new value
	 * @return the update statement
	 */
	Update<T> entry(Map<String, ?> entry);

	/**
	 * Sets the values to be updated by the update statement. The data is given as a
	 * map of element names of the target entity set to their new values.
	 *
	 * @param data the data to be updated as a map from element name to new value
	 * @return the update statement
	 */
	Update<T> data(Map<String, ?> data);

	/**
	 * Adds an element to be updated to the update statement.
	 * 
	 * @see #set(String, CqnValue) to update with an {@link CqnExpression
	 *      expression} value
	 *
	 * @param elementName the element name of the target entity
	 * @param value       the new value of the element
	 * @return the update statement
	 */
	Update<T> data(String elementName, Object value);

	/**
	 * Adds an element to be updated to the update statement. The update value can
	 * be a {@link CqnExpression}.
	 *
	 * @param <E>    the type of the referenced element
	 * @param ref the element ref of the target entity
	 * @param value       the new value of the element
	 * @return the update statement
	 */
	<E> Update<T> set(Function<T, ElementRef<E>> ref, Function<ElementRef<E>, Value<?>> value);

	/**
	 * Adds an element to be updated to the update statement. The update value can
	 * be a {@link CqnExpression}.
	 *
	 * @param elementName the element name of the target entity
	 * @param value       the new value of the element as {@link CqnValue}
	 * @return the update statement
	 */
	Update<T> set(String elementName, CqnValue value);

	/**
	 * Sets the values to be updated by the update statement. The setters are given
	 * as a map of element names of the target entity set to their new
	 * {@link CqnValue}, which can be a {@link CqnExpression}.
	 *
	 * @param setters the data to be updated as a map from element name to new
	 *                {@link CqnValue}
	 * @return the update statement
	 */
	Update<T> setters(Map<String, CqnValue> setters);

	/**
	 * Sets the where condition of this update statement to a given predicate. If
	 * this update statement already has a where condition it is replaced with the
	 * given.
	 *
	 * @param predicate a {@link CqnPredicate}
	 * @return the update statement
	 */
	@Override
	Update<T> where(CqnPredicate predicate);

	/**
	 * Sets the where condition of this update statement. The where condition is
	 * provided as a function that accepts a model object representing the entity
	 * targeted by the update and returning the where condition.
	 *
	 * @param predicate a {@link Function} providing the where condition
	 * @return the update statement
	 */
	@Override
	Update<T> where(Function<T, CqnPredicate> predicate);

	/**
	 * Sets the where condition of the update statement. The where condition is
	 * computed from a map of element names of the target entity set to their
	 * values, or a {@link CqnParameter}. The map entries are transformed into
	 * comparison predicates and joined via <b>and</b>.
	 *
	 * @param values the element name to value map defining the where condition
	 * @return the update statement
	 */
	@Override
	Update<T> matching(Map<String, ?> values);

	/**
	 * Sets the where condition of the update statement. The where condition is
	 * computed from the given vararg of element references of the target entity.
	 * For each element reference a predicate is created comparing the element ref
	 * with a {@link CqnParameter} that has the name of the ref. The predicates are
	 * joined via <b>and</b>.
	 *
	 * @param elementRefs the element references defining the where condition
	 * @return this update statement
	 */
	@Override
	Update<T> byParams(String... elementRefs);

	/**
	 * Sets the where condition of the update statement. The where condition is
	 * computed from the given collection of element references of the target
	 * entity. For each element ref a predicate is created comparing the ref with a
	 * {@link CqnParameter} that has the name of the ref. The predicates are joined
	 * via <b>and</b>.
	 *
	 * @param elementRefs the element references defining the where condition
	 * @return this update statement
	 */
	@Override
	Update<T> byParams(Collection<String> elementRefs);

	/**
	 * Sets the where condition of the update statement, requiring that the value of
	 * the single key element of the target entity set is equal to the given
	 * idValue.
	 *
	 * @param idValue the value of the target entity's key element
	 * @return the update statement
	 */
	@Override
	Update<T> byId(Object idValue);

}
