/*******************************************************************
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql.hana;

import com.google.common.annotations.Beta;
import com.sap.cds.Cds4jServiceLoader;
import com.sap.cds.ql.Source;
import com.sap.cds.ql.StructuredType;

@Beta
public interface HANA {

	HanaSupport hana = Cds4jServiceLoader.load(HanaSupport.class);

	/**
	 * Returns a function that generates a hierarchy based on recursive parent-child
	 * source data.
	 * 
	 * @param source the source that the hierarchy is to be generated from
	 */
	static Hierarchy hierarchy(Source<?> source) {
		return hana.hierarchy(source);
	}

	/**
	 * Returns a function to compute the subset of a given hierarchy, which contains
	 * all nodes that are ancestors of a of start nodes set
	 * 
	 * @param hierarchy the source hierarchy
	 * @return a hierarchy navigation function to compute the ancestors
	 */
	static HierarchySubset ancestors(Hierarchy hierarchy) {
		return hana.ancestors(hierarchy);
	}

	/**
	 * Returns a function to compute the subset of a given hierarchy, which contains
	 * all nodes that are ancestors of a start nodes set. The source is an entity,
	 * which containing all of the basic hierarchy attributes computed by a
	 * {@link #hierarchy} generation function
	 * 
	 * @param entity the entity
	 * @return a hierarchy navigation function to compute the ancestors
	 */
	static HierarchySubset ancestors(StructuredType<?> entity) {
		return hana.ancestors(entity);
	}

	/**
	 * Returns a function to compute the subset of a given hierarchy, which contains
	 * all nodes that are descendants of a set of start nodes
	 * 
	 * @param hierarchy the source hierarchy
	 * @return a hierarchy navigation function to compute the descendants
	 */
	static HierarchySubset descendants(Hierarchy hierarchy) {
		return hana.descendants(hierarchy);
	}

	/**
	 * Returns a function to compute the subset of a given hierarchy, which contains
	 * all nodes that are descendants of a set of start nodes
	 * 
	 * @param entity the source hierarchy
	 * @return a hierarchy navigation function to compute the descendants
	 */
	static HierarchySubset descendants(StructuredType<?> entity) {
		return hana.descendants(entity);
	}
	
}
