/************************************************************************
 * © 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.ql;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toSet;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.sap.cds.ql.cqn.CqnCaseExpression;
import com.sap.cds.ql.cqn.CqnListValue;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelectListItem;

/**
 * Builder to define sets of structured CDS entities and types using filtered
 * path expressions, to define predicates on these sets, and to address elements
 * of structured types.
 *
 * Note that some methods in this interface mutually exclude each other and
 * cannot be combined. For example, the methods {@link #filter(CqnPredicate)
 * filter}, {@link #filterByParams(String...) filterByParams}, and
 * {@link #matching(Map) matching} all set the where condition of the enclosing
 * statement, overwriting any previously stored conditions.
 */
public interface StructuredType<T extends StructuredType<T>> extends Source<T>, Selectable {

	StructuredType<?> to(String path);

	<E extends StructuredType<E>> E to(String path, Class<E> type);

	<E> ElementRef<E> get(String path);

	<E> ElementRef<E> get(String path, Class<E> type);

	T filter(CqnPredicate p);

	T filter(Function<T, CqnPredicate> p);

	/**
	 * Creates a predicate that tests whether <em>any</em> instance of the entity
	 * set referenced by this structured type matches a given condition. The filter
	 * condition is provided as a function that accepts a model object representing
	 * the entity referenced by this structured type and returning the filter
	 * condition.
	 *
	 * @param p the condition to test for
	 * @return the match predicate
	 */
	Predicate anyMatch(Function<T, CqnPredicate> p);

	/**
	 * Creates a predicate that tests whether <em>any</em> instance of the entity
	 * set referenced by this structured type matches a given condition.
	 *
	 * @param p the condition to test for
	 * @return the match predicate
	 */
	Predicate anyMatch(CqnPredicate p);

	/**
	 * Creates a predicate that tests whether <em>all</em> elements of the entity
	 * set referenced by this structured type match a given condition. The filter
	 * condition is provided as a function that accepts a model object representing
	 * the entity referenced by this structured type and returning the filter
	 * condition.
	 *
	 * @param p the condition to test for
	 * @return the match predicate
	 */
	Predicate allMatch(Function<T, CqnPredicate> p);

	/**
	 * Creates a predicate that tests whether <em>all</em> elements of the entity
	 * set referenced by this structured type match a given condition.
	 *
	 * @param p the condition to test for
	 * @return the match predicate
	 */
	Predicate allMatch(CqnPredicate p);

	/**
	 * Creates a predicate that tests whether <em>any</em> instance of the entity
	 * set referenced by this structured type exists, i.e. the entity set referenced
	 * by this structured type is not empty.
	 *
	 * @return the exists predicate
	 */
	Predicate exists();

	Predicate eTag(Object value);

	Predicate eTag(CqnListValue values);

	T matching(Map<String, ?> values);

	default T filterByParams(String... elementRefs) {
		return filterByParams(asList(elementRefs).stream().collect(toSet()));
	}

	T filterByParams(Collection<String> elementRefs);

	CqnSelectListItem inline();

	default CqnSelectListItem inline(Function<T, ? extends Selectable> item) {
		return inline(Collections.singletonList(item));
	}

	default CqnSelectListItem inline(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2) {
		return inline(Arrays.asList(item1, item2));
	}

	default CqnSelectListItem inline(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3) {
		return inline(Arrays.asList(item1, item2, item3));
	}

	default CqnSelectListItem inline(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3, Function<T, ? extends Selectable> item4) {
		return inline(Arrays.asList(item1, item2, item3, item4));
	}

	default CqnSelectListItem inline(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3, Function<T, ? extends Selectable> item4,
			Function<T, ? extends Selectable> item5) {
		return inline(Arrays.asList(item1, item2, item3, item4, item5));
	}

	default CqnSelectListItem inline(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3, Function<T, ? extends Selectable> item4,
			Function<T, ? extends Selectable> item5, Function<T, ? extends Selectable> item6) {
		return inline(Arrays.asList(item1, item2, item3, item4, item5, item6));
	}

	default CqnSelectListItem inline(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3, Function<T, ? extends Selectable> item4,
			Function<T, ? extends Selectable> item5, Function<T, ? extends Selectable> item6,
			Function<T, ? extends Selectable> item7) {
		return inline(Arrays.asList(item1, item2, item3, item4, item5, item6, item7));
	}

	@SuppressWarnings("unchecked")
	default CqnSelectListItem inline(Function<T, ? extends Selectable>... items) {
		return inline(Arrays.asList(items));
	}

	CqnSelectListItem inline(List<Function<T, ? extends Selectable>> items);

	CqnSelectListItem inline(Iterable<? extends Selectable> items);

	CqnSelectListItem inline(Selectable... items);

	CqnSelectListItem inline(String... refs);

	Expand<T> expand();

	default Expand<T> expand(Function<T, ? extends Selectable> item) {
		return expand(Collections.singletonList(item));
	}

	default Expand<T> expand(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2) {
		return expand(Arrays.asList(item1, item2));
	}

	default Expand<T> expand(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3) {
		return expand(Arrays.asList(item1, item2, item3));
	}

	default Expand<T> expand(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3, Function<T, ? extends Selectable> item4) {
		return expand(Arrays.asList(item1, item2, item3, item4));
	}

	default Expand<T> expand(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3, Function<T, ? extends Selectable> item4,
			Function<T, ? extends Selectable> item5) {
		return expand(Arrays.asList(item1, item2, item3, item4, item5));
	}

	default Expand<T> expand(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3, Function<T, ? extends Selectable> item4,
			Function<T, ? extends Selectable> item5, Function<T, ? extends Selectable> item6) {
		return expand(Arrays.asList(item1, item2, item3, item4, item5, item6));
	}

	default Expand<T> expand(Function<T, ? extends Selectable> item1, Function<T, ? extends Selectable> item2,
			Function<T, ? extends Selectable> item3, Function<T, ? extends Selectable> item4,
			Function<T, ? extends Selectable> item5, Function<T, ? extends Selectable> item6,
			Function<T, ? extends Selectable> item7) {
		return expand(Arrays.asList(item1, item2, item3, item4, item5, item6, item7));
	}

	@SuppressWarnings("unchecked")
	default Expand<T> expand(Function<T, ? extends Selectable>... items) {
		return expand(Arrays.asList(items));
	}

	Expand<T> expand(List<Function<T, ? extends Selectable>> items);

	Expand<T> expand(Iterable<? extends Selectable> items);

	Expand<T> expand(Selectable... items);

	Expand<T> expand(String... refs);

	CqnSelectListItem _all();

	Predicate exists(Function<T, Select<?>> subQuery);

	/**
	 * Returns an immutable ref.
	 *
	 * @return the {@link StructuredTypeRef}
	 */
	@Override
	StructuredTypeRef asRef();

	/**
	 * Gives an alias to this structured type.
	 *
	 * @param alias the alias or null
	 * @return this structured type
	 */
	StructuredType<T> as(String alias);

	/**
	 * Starts a chain of when-then expressions.
	 * 
	 * @param pred the {@link CqnCaseExpression.Case#condition() condition} of the
	 *             first case.
	 * @return a builder to specify the case's return value
	 */
	When when(CqnPredicate pred);

}
