/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql.cqn;

import java.util.Map;

import com.sap.cds.reflect.CdsEntity;

/**
 * Contains the analysis result for a {@link CqnStructuredTypeRef} or
 * {@link CqnStatement}.
 */
public interface AnalysisResult extends Path {

	/**
	 * Returns the {@link CdsEntity} that is the root of the {@code ref}.
	 *
	 * @return the root {@code CdsEntity}
	 */
	CdsEntity rootEntity();

	/**
	 * Returns the element name to value mapping for the key elements of the
	 * {@link CdsEntity} that is the root of the {@code ref}.
	 * 
	 * Extracts the key values, i.e. the values of the key elements of the entity
	 * that is the root of the {@code ref}. If the filter does not uniquely restrict
	 * a key element to a particular value, the element's name is mapped to
	 * {@code null}.
	 *
	 * @return a map relating all key element names to key values
	 */
	Map<String, Object> rootKeys();

	/**
	 * Returns the element name to value mapping for the key elements of the
	 * {@link CdsEntity} that is targeted by the {@code ref}.
	 * 
	 * Extracts the key values, i.e. the values of the key elements to the entity
	 * targeted by the {@code ref} and optionally the statement's {@code where}
	 * condition. If the combined filter does not uniquely restrict a key element to
	 * a particular value, the element's name is mapped to {@code null}.
	 *
	 * @return a map relating all key element names to key values
	 */
	Map<String, Object> targetKeys();

	/**
	 * Returns the element name to value mapping for the key elements of the
	 * {@link CdsEntity} that is targeted by the {@code ref}.
	 * 
	 * Extracts the key values, i.e. the values of the key elements to the entity
	 * targeted by the {@code ref} and optionally the statement's {@code where}
	 * condition. If the combined filter does not uniquely restrict a key element to
	 * a particular value, the key name is not present in the returned map.
	 *
	 * @return a map relating key element names to key values
	 */
	Map<String, Object> targetKeyValues();

	/**
	 * Returns the element name to value mapping for all elements of the
	 * {@link CdsEntity} that is targeted by the {@code ref}.
	 * 
	 * Extracts the values, i.e. the values of the elements to the entity targeted
	 * by the {@code ref} and optionally the statement's {@code where} condition. If
	 * the combined filter does not uniquely restrict an element to a particular
	 * value, the element's name is not present in the returned map.
	 *
	 * @return a map relating all element names to values
	 */
	Map<String, Object> targetValues();

	/**
	 * Returns the {@link CdsEntity} targeted by the {@code ref}.
	 *
	 * @return the target {@code CdsEntity}
	 */
	CdsEntity targetEntity();

}
