/************************************************************************
 * © 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.ql;

import java.util.Collection;
import java.util.Map;
import java.util.function.Function;

import com.sap.cds.ql.cqn.CqnFilterableStatement;
import com.sap.cds.ql.cqn.CqnParameter;
import com.sap.cds.ql.cqn.CqnPredicate;

public interface FilterableStatement<T, S extends FilterableStatement<T, ?>>
		extends Statement<S>, CqnFilterableStatement {

	/**
	 * Sets the where condition of this filterable statement to a given predicate.
	 * If this filterable statement already has a where condition it is replaced
	 * with the given.
	 *
	 * @param predicate a {@link CqnPredicate}
	 * @return the filterable statement
	 */
	S where(CqnPredicate predicate);

	/**
	 * Sets the where condition of this filterable statement. The where condition is
	 * provided as a function that accepts a model object representing the entity
	 * targeted by the filterable and returning the where condition.
	 *
	 * @param predicate a {@link Function} providing the where condition
	 * @return the filterable statement
	 */
	S where(Function<T, CqnPredicate> predicate);

	/**
	 * Sets the where condition of the filterable statement. The where condition is
	 * computed from a map of element names of the target entity set to their
	 * values, or a {@link CqnParameter}. The map entries are transformed into
	 * comparison predicates and joined via <b>and</b>.
	 *
	 * @param values the element name to value map defining the where condition
	 * @return the filterable statement
	 */
	S matching(Map<String, ?> values);

	/**
	 * Sets the where condition of the filterable statement. The where condition is
	 * computed from the given vararg of element references of the target entity.
	 * For each element reference a predicate is created comparing the element ref
	 * with a {@link CqnParameter} that has the name of the ref. The predicates are
	 * joined via <b>and</b>.
	 *
	 * @param elementRefs the element references defining the where condition
	 * @return this filterable statement
	 */
	S byParams(String... elementRefs);

	/**
	 * Sets the where condition of the filterable statement. The where condition is
	 * computed from the given collection of element references of the target
	 * entity. For each element ref a predicate is created comparing the ref with a
	 * {@link CqnParameter} that has the name of the ref. The predicates are joined
	 * via <b>and</b>.
	 *
	 * @param elementRefs the element references defining the where condition
	 * @return this filterable statement
	 */
	S byParams(Collection<String> elementRefs);

	/**
	 * Sets the where condition of the filterable statement, requiring that the
	 * value of the single key element of the target entity set is equal to the
	 * given idValue.
	 *
	 * @param idValue the value of the target entity's key element
	 * @return the filterable statement
	 */
	S byId(Object idValue);

}
