/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds;

import com.google.common.annotations.Beta;
import com.sap.cds.reflect.CdsDefinitionNotFoundException;
import com.sap.cds.reflect.CdsElementNotFoundException;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import java.util.List;
import java.util.Map;

/**
 * Converts JSON data to and from CDS (Core Data Services) data structures.
 *
 * <p>Provides methods for serializing and deserializing JSON objects and arrays, as well as
 * validating data against CDS types.
 */
@Beta
public interface CdsJsonConverter {

  /**
   * Parses a JSON object string into {@link CdsData} using the given CDS structured type.
   *
   * @param jsonObject JSON object as string (not null)
   * @param type CDS structured type (not null)
   * @return parsed {@link CdsData}
   * @throws CdsJsonConverterException if parsing or mapping fails
   */
  CdsData fromJsonObject(String jsonObject, CdsStructuredType type);

  /**
   * Parses a JSON object string into the given {@link CdsData} subtype.
   *
   * @param <T> target CDS data type
   * @param jsonObject JSON object as string (not null)
   * @param type target class (not null)
   * @return parsed instance of {@code type}
   * @throws CdsJsonConverterException if parsing or mapping fails
   * @throws CdsDefinitionNotFoundException if the CDS type is unknown
   */
  <T extends CdsData> T fromJsonObject(String jsonObject, Class<T> type);

  /**
   * Parses a JSON array string into a list of {@link CdsData} using the given CDS structured type.
   *
   * @param jsonArray JSON array as string (not null)
   * @param type CDS structured type (not null)
   * @return list of parsed {@link CdsData}
   * @throws CdsJsonConverterException if parsing or mapping fails
   */
  List<CdsData> fromJsonArray(String jsonArray, CdsStructuredType type);

  /**
   * Parses a JSON array string into a list of the given {@link CdsData} subtype.
   *
   * @param <T> target CDS data type
   * @param jsonArray JSON array as string (not null)
   * @param type target class (not null)
   * @return list of parsed instances of {@code type}
   * @throws CdsJsonConverterException if parsing or mapping fails
   * @throws CdsDefinitionNotFoundException if the CDS type is unknown
   */
  <T extends CdsData> List<T> fromJsonArray(String jsonArray, Class<T> type);

  /**
   * Serializes a {@link Map} instance to a JSON object string.
   *
   * @param data map instance to serialize (not null)
   * @return JSON object as string
   * @throws CdsJsonConverterException if serialization fails
   */
  String toJson(Map<String, Object> data);

  /**
   * Serializes a list of {@link Map} instances to a JSON array string.
   *
   * @param data list of instances to serialize (not null)
   * @return JSON array as string
   * @throws CdsJsonConverterException if serialization fails
   */
  String toJson(List<? extends Map<String, Object>> data);

  /**
   * Creates a new builder for configuring and building a {@link CdsJsonConverter}.
   *
   * @param model the {@link CdsModel} to be used for validation during conversion
   * @return a new {@link Builder} instance
   * @throws CdsJsonConverterException in case of an error
   */
  static CdsJsonConverter.Builder builder(CdsModel model) {
    return factory.createBuilder(model);
  }

  /** Defines how to handle JSON properties not declared in the CDS type. */
  public enum UnknownPropertyHandling {

    /** Attempt to infer type and include undeclared properties. */
    INCLUDE,

    /** Reject undeclared properties with an {@link CdsElementNotFoundException}. */
    REJECT,

    /** Ignore undeclared properties. */
    IGNORE
  }

  /** Enum defining the serialization of CDS Decimal types to JSON. */
  enum DecimalSerialization {
    /** Writes all CDS Decimal types as numbers in JSON format. This is the default setting. */
    NUMBER,
    /** Writes all CDS Decimal types as strings in JSON format without an exponent field. */
    STRING
  }

  /** Builder interface for configuring and building a {@link CdsJsonConverter} instance. */
  interface Builder {

    /**
     * Sets the strategy for handling undeclared JSON properties.
     *
     * @param handling the strategy to use
     * @return this builder instance
     */
    CdsJsonConverter.Builder unknownPropertyHandling(UnknownPropertyHandling handling);

    /**
     * Sets how CDS Decimal types serialized to JSON.
     *
     * @param decimalSerialization the serialization
     * @return this builder instance
     */
    CdsJsonConverter.Builder decimalSerialization(DecimalSerialization decimalSerialization);

    /**
     * Builds the configured {@link CdsJsonConverter} instance.
     *
     * @return a new {@link CdsJsonConverter} instance
     */
    CdsJsonConverter build();
  }

  /** Factory interface for creating {@link Builder} instances. */
  interface Factory {

    /**
     * Creates a new {@link Builder} for the specified CDS model.
     *
     * @param model the {@link CdsModel} to use for validation
     * @return a new {@link Builder} instance
     */
    CdsJsonConverter.Builder createBuilder(CdsModel model);
  }

  /** The factory instance for creating {@link Builder} instances. */
  Factory factory = Cds4jServiceLoader.load(Factory.class);
}
