/*
 * © 2023 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql;

import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference;
import java.util.List;
import java.util.stream.Stream;

/**
 * Builder for CQN {@link CqnReference references}.
 *
 * <p>The {@link RefSegment segments} of the ref can be modified in-place. Call {@link #build} to
 * create an immutable ref.
 */
public interface RefBuilder<T extends CqnReference> {

  /**
   * Returns the segments of this ref builder as {@link List}.
   *
   * @return the list of {@link RefSegment RefSegments}
   */
  List<RefSegment> segments();

  /**
   * Returns the segments of this ref builder as {@link Stream}.
   *
   * @return the stream of {@link RefSegment RefSegments}
   */
  Stream<RefSegment> stream();

  /**
   * Returns the first segment of this ref builder.
   *
   * @return the first {@link RefSegment}
   */
  RefSegment rootSegment();

  /**
   * Returns the last segment of this ref builder.
   *
   * @return the last {@link RefSegment}
   */
  RefSegment targetSegment();

  /**
   * Sets the alias of this ref builder.
   *
   * @param alias the alias
   * @return this ref builder
   */
  RefBuilder<T> as(String alias);

  /**
   * Sets the type of this ref builder.
   *
   * @param typeName the type name
   * @return this ref builder
   */
  RefBuilder<T> type(String typeName);

  /**
   * Creates an immutable ref
   *
   * @return an immutable ref of type T
   */
  T build();

  /** A modifiable ref segment. */
  interface RefSegment extends CqnReference.Segment {

    /**
     * Sets the id of this segment.
     *
     * @param id the new segment id
     * @return this ref segment
     */
    RefSegment id(String id);

    /**
     * Sets the filter of this segment.
     *
     * @param filter the new segment filter
     * @return this ref segment
     */
    RefSegment filter(CqnPredicate filter);
  }
}
