/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql.cqn;

import java.util.List;

/**
 * A case expression, which represents IF - THEN - ELSE logic. The result value is determined by
 * evaluating a list of {@link Case cases}. If a {@link Case#condition()} of a case evaluates to
 * TRUE the {@link Case#value() result value} of the case is returned. If no condition of any case
 * evaluates to TRUE the {@link #defaultValue() default value} is returned.
 */
public interface CqnCaseExpression extends CqnExpression {
  /**
   * The {@link Case cases}, which are evaluated.
   *
   * @return the cases
   */
  List<Case> cases();

  /** A case, which relates a condition to a result value. */
  interface Case {
    /**
     * The condition
     *
     * @return the condition
     */
    CqnPredicate condition();

    /**
     * The result value
     *
     * @return the result value
     */
    CqnValue value();
  }

  /**
   * The default value, which is returned if no condition of any case matches
   *
   * @return the default value
   */
  CqnValue defaultValue();

  @Override
  default boolean isCaseExpression() {
    return true;
  }

  @Override
  default CqnCaseExpression asCaseExpression() {
    return this;
  }

  @Override
  default void accept(CqnVisitor visitor) {
    for (Case c : cases()) {
      c.condition().accept(visitor);
      c.value().accept(visitor);
    }

    defaultValue().accept(visitor);

    visitor.visit(this);
  }
}
