/*
 * © 2018-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql.cqn;

import java.util.Collection;

/**
 * The {@code in} predicate tests wether an {@link #value() actual value} is equal to any value from
 * a defined {@link #valueSet() value set}.
 *
 * <p>It evaluates to {@code false} if the value set is empty.
 */
public interface CqnInPredicate extends CqnPredicate {

  /**
   * The actual {@link CqnValue} that is tested against the values given by {@link #valueSet()}.
   *
   * @return the actual {@link CqnValue}
   */
  CqnValue value();

  /**
   * Returns the collection of expected values in case {@link #valueSet()} is a {@link
   * CqnListValue}, otherwise a {@link ClassCastException} is thrown.
   *
   * @throws ClassCastException if this value is not a {@link CqnListValue}
   * @return the expected values
   */
  default Collection<CqnValue> values() {
    return valueSet().asList().values().toList();
  }

  /**
   * Defines the set of expected values the {@link #value() actual value} is tested against. The
   * returned {@link CqnValue} can either be a {@link CqnListValue} containing all expected values,
   * a {@link CqnReference} pointing to a collection-valued element, or a {@link CqnParameter}
   * pointing to a collection of values.
   *
   * @return the set of expected values
   */
  CqnValue valueSet();

  @Override
  default void accept(CqnVisitor visitor) {
    value().accept(visitor);
    valueSet().accept(visitor);

    visitor.visit(this);
  }
}
