/*
 * © 2018-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql.cqn;

import com.google.common.annotations.Beta;
import com.sap.cds.CqnTableFunction;

public interface CqnSource extends CqnToken {

  /**
   * Returns {@code true} if this is a {@link CqnStructuredTypeRef}.
   *
   * @return {@code true} if this is a structured type reference, otherwise {@code false}
   */
  default boolean isRef() {
    return false;
  }

  /**
   * Returns {@code true} if this is a {@link CqnSelect} statement.
   *
   * @return {@code true} if this is a select statement, otherwise {@code false}
   */
  default boolean isSelect() {
    return false;
  }

  /**
   * Returns {@code true} if this is a {@link CqnJoin}.
   *
   * @return {@code true} if this is a join, otherwise {@code false}
   */
  default boolean isJoin() {
    return false;
  }

  /**
   * Returns {@code true} if this is a {@link CqnTableFunction}.
   *
   * @return {@code true} if this is a table function, otherwise {@code false}
   */
  @Beta
  default boolean isTableFunction() {
    return false;
  }

  /**
   * Casts this CQN source to {@code CqnSelect}.
   *
   * @throws ClassCastException if this CQN source is not a select
   * @return this CQN source as a CqnSelect
   */
  default CqnStructuredTypeRef asRef() {
    throw new ClassCastException("Cannot cast to CqnStructuredTypeRef");
  }

  /**
   * Casts this CQN source to a {@code CqnSelect}.
   *
   * @throws ClassCastException if this CQN source is not a select
   * @return this CQN source as a CqnSelect
   */
  default CqnSelect asSelect() {
    throw new ClassCastException("Cannot cast to CqnSelect");
  }

  /**
   * Casts this CQN source to a {@code CqnJoin}.
   *
   * @throws ClassCastException if this CQN source is not a join
   * @return this CQN source as a CqnJoin
   */
  default CqnJoin asJoin() {
    throw new ClassCastException("Cannot cast to CqnJoin");
  }

  /**
   * Casts this CQN source to a {@code CqnTableFunction}.
   *
   * @throws ClassCastException if this CQN source is no table function
   * @return this CQN source as a CqnTableFunction
   */
  @Beta
  default CqnTableFunction asTableFunction() {
    throw new ClassCastException("Cannot cast to CqnTableFunction");
  }
}
