/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql.cqn.transformation;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.cqn.CqnToken;

/**
 * A {@link CqnToken} tree visitor conform to the visitor design pattern. Classes implementing this
 * interface operate on a tree of {@code CqnTransformation}
 *
 * <p>When a visitor is passed as an argument to a transformation's {@code accept} method, the
 * transformation's {@code accept} method of the children is NOT called. If the deep-first behavior
 * is required it must be implemented manually.
 */
@Beta
public interface CqnTransformationVisitor {

  /**
   * Called for each traversed {@link CqnAggregateTransformation} transformation
   *
   * @param aggregate the aggregate transformation
   */
  default void visit(CqnAggregateTransformation aggregate) {}

  /**
   * Called for each traversed {@link CqnAncestorsTransformation} transformation
   *
   * @param ancestors the ancestors transformation
   */
  default void visit(CqnAncestorsTransformation ancestors) {
    visit((CqnHierarchySubsetTransformation) ancestors);
  }

  /**
   * Called for each traversed {@link CqnComputeTransformation} transformation
   *
   * @param compute the compute transformation
   */
  default void visit(CqnComputeTransformation compute) {}

  /**
   * Called for each traversed {@link CqnCustomTransformation} transformation
   *
   * @param custom the custom transformation
   */
  default void visit(CqnCustomTransformation custom) {}

  /**
   * Called for each traversed {@link CqnDescendantsTransformation} transformation
   *
   * @param descendants the descendants transformation
   */
  default void visit(CqnDescendantsTransformation descendants) {
    visit((CqnHierarchySubsetTransformation) descendants);
  }

  /**
   * Called for each traversed {@link CqnFilterTransformation} transformation
   *
   * @param filter the filter transformation
   */
  default void visit(CqnFilterTransformation filter) {}

  /**
   * Called for each traversed {@link CqnGroupByTransformation} transformation
   *
   * @param groupBy the group by transformation
   */
  default void visit(CqnGroupByTransformation groupBy) {}

  /**
   * Called for each traversed {@link CqnOrderByTransformation} transformation
   *
   * @param orderBy the order by transformation
   */
  default void visit(CqnOrderByTransformation orderBy) {}

  /**
   * Called for each traversed {@link CqnSearchTransformation} transformation
   *
   * @param search the search transformation
   */
  default void visit(CqnSearchTransformation search) {}

  /**
   * Called for each traversed {@link CqnTopTransformation} transformation
   *
   * @param top the top transformation
   */
  default void visit(CqnTopTransformation top) {}

  /**
   * Called for each traversed {@link CqnSkipTransformation} transformation
   *
   * @param skip the skip transformation
   */
  default void visit(CqnSkipTransformation skip) {}

  /**
   * Called for each traversed {@link CqnTopLevelsTransformation} transformation
   *
   * @param topLevels the top levels (hierarchy generator) transformation
   */
  default void visit(CqnTopLevelsTransformation topLevels) {}

  /**
   * Called for each traversed {@link CqnHierarchySubsetTransformation} transformation: ancestors
   * and descendants
   *
   * @param hierarchySubset the subset (hierarchy navigation) transformation
   */
  default void visit(CqnHierarchySubsetTransformation hierarchySubset) {
    visit((CqnHierarchyTransformation) hierarchySubset);
  }

  /**
   * Called for each traversed {@link CqnHierarchyTransformation} transformation: ancestors,
   * descendants and topLevels
   *
   * @param hierarchy the hierarchy transformation
   */
  default void visit(CqnHierarchyTransformation hierarchy) {}
}
