/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.reflect;

import static com.sap.cds.Cds4jServiceLoader.find;

import com.sap.cds.CdsException;
import java.io.InputStream;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * The CdsModel represents the complete CDS model and is the starting point for the introspection.
 */
public interface CdsModel {

  Optional<Reader> reader = find(CdsModel.Reader.class);

  static CdsModel.Reader reader() {
    return reader.orElseThrow(
        () -> new CdsException("Cannot find implementation for CDS Model Reader"));
  }

  /**
   * Creates a CDS model from a CSN input stream.
   *
   * @param csn the CSN input stream
   * @return the CDS model
   */
  static CdsModel read(InputStream csn) {
    return reader().readCsn(csn);
  }

  /**
   * Creates a CDS model from a CSN String.
   *
   * @param csn the CSN input as String
   * @return the CDS model
   */
  static CdsModel read(String csn) {
    return reader().readCsn(csn);
  }

  /**
   * Returns a sequential {@code Stream} over all {@link CdsAnnotation CdsAnnotation(s)} for the
   * given namespace.
   *
   * @param namespace the namespace
   * @return a sequential {@code Stream} over {@code CdsAnnotation(s)} for the given namespace
   */
  Stream<CdsAnnotation<?>> annotations(String namespace);

  /**
   * Returns a sequential {@code Stream} over all {@link CdsService} definitions in this model.
   *
   * @return a sequential {@code Stream} over the {@link CdsService} definitions
   */
  Stream<CdsService> services();

  /**
   * Get a {@link CdsService} via its fully qualified name.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsService}
   * @return the {@code CdsService} with the given fully qualified name
   * @throws CdsDefinitionNotFoundException if there is no {@code CdsService} with the given name
   */
  CdsService getService(String qualifiedName);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsService}.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsService}
   * @return an {@code Optional} describing the {@code CdsService} with the given qualified name, or
   *     an empty {@code Optional} if there is no {@code CdsService} with this name
   */
  Optional<CdsService> findService(String qualifiedName);

  /**
   * Returns a sequential {@code Stream} over all {@link CdsType} definitions in this model.
   *
   * @return a sequential {@code Stream} over the {@link CdsType} definitions
   */
  Stream<CdsType> types();

  /**
   * Get a {@link CdsType} via its fully qualified name.
   *
   * @param <T> the return type
   * @param qualifiedName the fully qualified name of the {@code CdsType}
   * @return the {@code CdsType} with the given fully qualified name
   * @throws CdsDefinitionNotFoundException if there is no {@code CdsType} with the given name
   */
  <T extends CdsType> T getType(String qualifiedName);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsType}.
   *
   * @param <T> the return type
   * @param qualifiedName the fully qualified name of the {@code CdsType}
   * @return an {@code Optional} describing the {@code CdsType} with the given qualified name, or an
   *     empty {@code Optional} if there is no {@code CdsType} with this name
   */
  <T extends CdsType> Optional<T> findType(String qualifiedName);

  /**
   * Returns a sequential {@code Stream} over all {@link CdsEntity} definitions in this model.
   *
   * @return a sequential {@code Stream} over the {@link CdsEntity} definitions
   */
  Stream<CdsEntity> entities();

  /**
   * Returns a sequential {@code Stream} over all non abstract {@link CdsEntity} definitions in this
   * model.
   *
   * @return a sequential {@code Stream} over the non abstract {@link CdsEntity} definitions
   */
  default Stream<CdsEntity> concreteEntities() {
    return entities().filter(e -> !e.isAbstract());
  }

  /**
   * Get a {@link CdsEntity} via its fully qualified name.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsEntity}
   * @return the {@code CdsEntity} with the given fully qualified name
   * @throws CdsDefinitionNotFoundException if there is no {@code CdsEntity} with the given name
   */
  CdsEntity getEntity(String qualifiedName);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsEntity}.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsEntity}
   * @return an {@code Optional} describing the {@code CdsEntity} with the given qualified name, or
   *     an empty {@code Optional} if there is no {@code CdsEntity} with this name
   */
  Optional<CdsEntity> findEntity(String qualifiedName);

  /**
   * Returns a sequential {@code Stream} over all {@link CdsAction} definitions in this model.
   *
   * @return a sequential {@code Stream} over the {@link CdsAction} definitions
   */
  Stream<CdsAction> actions();

  /**
   * Get a {@link CdsAction} via its fully qualified name.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsAction}
   * @return the {@code CdsAction} with the given fully qualified name
   * @throws CdsDefinitionNotFoundException if there is no {@code CdsAction} with the given name
   */
  CdsAction getAction(String qualifiedName);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsAction}.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsAction}
   * @return an {@code Optional} describing the {@code CdsAction} with the given qualified name, or
   *     an empty {@code Optional} if there is no {@code CdsAction} with this name
   */
  Optional<CdsAction> findAction(String qualifiedName);

  /**
   * Returns a sequential {@code Stream} over all {@link CdsFunction} definitions in this model.
   *
   * @return a sequential {@code Stream} over the {@link CdsFunction} definitions
   */
  Stream<CdsFunction> functions();

  /**
   * Get a {@link CdsFunction} via its fully qualified name.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsFunction}
   * @return the {@code CdsFunction} with the given fully qualified name
   * @throws CdsDefinitionNotFoundException if there is no {@code CdsFunction} with the given name
   */
  CdsFunction getFunction(String qualifiedName);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsFunction}.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsFunction}
   * @return an {@code Optional} describing the {@code CdsFunction} with the given qualified name,
   *     or an empty {@code Optional} if there is no {@code CdsFunction} with this name
   */
  Optional<CdsFunction> findFunction(String qualifiedName);

  /**
   * Returns a sequential {@code Stream} over all {@link CdsEvent} definitions in this model.
   *
   * @return a sequential {@code Stream} over the {@link CdsEvent} definitions
   */
  Stream<CdsEvent> events();

  /**
   * Get a {@link CdsEvent} via its fully qualified name.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsEvent}
   * @return the {@code CdsEvent} with the given fully qualified name
   * @throws CdsDefinitionNotFoundException if there is no {@code CdsEvent} with the given name
   */
  CdsEvent getEvent(String qualifiedName);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsEvent}.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsEvent}
   * @return an {@code Optional} describing the {@code CdsEvent} with the given qualified name, or
   *     an empty {@code Optional} if there is no {@code CdsEvent} with this name
   */
  Optional<CdsEvent> findEvent(String qualifiedName);

  /**
   * Returns a sequential {@code Stream} over all {@link CdsStructuredType} definitions in this
   * model.
   *
   * @return a sequential {@code Stream} over the {@link CdsStructuredType} definitions
   */
  Stream<CdsStructuredType> structuredTypes();

  /**
   * Get a {@link CdsStructuredType} via its fully qualified name.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsStructuredType}
   * @return the {@code CdsStructuredType} with the given fully qualified name
   * @throws CdsDefinitionNotFoundException if there is no {@code CdsStructuredType} with the given
   *     name
   */
  CdsStructuredType getStructuredType(String qualifiedName);

  /**
   * Returns an {@link Optional} wrapping a {@link CdsStructuredType}.
   *
   * @param qualifiedName the fully qualified name of the {@code CdsStructuredType}
   * @return an {@code Optional} describing the {@code CdsStructuredType} with the given qualified
   *     name, or an empty {@code Optional} if there is no {@code CdsStructuredType} with this name
   */
  Optional<CdsStructuredType> findStructuredType(String qualifiedName);

  /**
   * Returns a sequential {@code Stream} over all aspect definitions in this model.
   *
   * @return a sequential {@code Stream} over the aspect definitions
   */
  Stream<CdsStructuredType> aspects();

  /**
   * Get an aspect definition via its fully qualified name.
   *
   * @param qualifiedName the fully qualified name of the aspect definition
   * @return the aspect definition with the given fully qualified name
   * @throws CdsDefinitionNotFoundException if there is no aspect definition with the given name
   */
  CdsStructuredType getAspect(String qualifiedName);

  /**
   * Returns an {@link Optional} wrapping an aspect definition.
   *
   * @param qualifiedName the fully qualified name of the aspect definition
   * @return an {@code Optional} describing the aspect definition with the given qualified name, or
   *     an empty {@code Optional} if there is no aspect definition with this name
   */
  Optional<CdsStructuredType> findAspect(String qualifiedName);

  /**
   * Returns the meta property for the given key.
   *
   * @param <T> the expected type of the meta property
   * @param key the qualified name of the meta property
   * @return the value of the meta property or {@code null} if there is no property with the given
   *     key
   */
  <T> T getMeta(String key);

  /**
   * Returns the CSN version this model was read from.
   *
   * @return the CSN version
   */
  String getVersion();

  /**
   * Accepts a {@link CdsVisitor} visiting the definitions in this model (depth-first)
   *
   * @param visitor the {@code CdsVisitor}
   */
  default void accept(CdsVisitor visitor) {
    services().forEach(s -> s.accept(visitor));
    types().forEach(t -> t.accept(visitor));
    entities().forEach(e -> e.accept(visitor));
    events().forEach(e -> e.accept(visitor));
    actions().forEach(e -> e.accept(visitor));
    functions().forEach(e -> e.accept(visitor));

    visitor.visit(this);
  }

  interface Reader {

    CdsModel readCsn(InputStream csn);

    CdsModel readCsn(String csn);
  }
}
