/*
 * © 2018-2022 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds;

import static com.sap.cds.Cds4jServiceLoader.load;

import com.sap.cds.connector.CdsDataStoreConnectorBuilder;
import com.sap.cds.connector.CdsDataStoreConnectorFactory;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.transaction.TransactionManager;

public interface CdsDataStoreConnector {

  /**
   * Create an instance of {@link CdsDataStoreConnectorBuilder} to construct the {@link
   * CdsDataStoreConnector} object.
   *
   * @param cdsModel the CDS Model
   * @param transactionManager transaction manager
   * @return the {@code CdsDataStoreConnectorBuilder}
   */
  static CdsDataStoreConnectorBuilder createJdbcConnector(
      CdsModel cdsModel, TransactionManager transactionManager) {
    CdsDataStoreConnectorFactory factory = load(CdsDataStoreConnectorFactory.class);
    return factory.create(cdsModel, transactionManager);
  }

  /**
   * Returns the {@link CdsDataStore}.
   *
   * @return the {@code CdsDataStore}
   */
  CdsDataStore connect();

  /**
   * Returns the {@link CdsDataStore}.
   *
   * @param session the session context object
   * @return the {@code CdsDataStore}
   */
  CdsDataStore connect(SessionContext session);

  /**
   * Returns the {@link CdsModel}.
   *
   * @return the {@code CdsModel}
   */
  CdsModel reflect();

  Capabilities capabilities();

  interface Capabilities {

    Capabilities ALL =
        new Capabilities() {
          @Override
          public boolean supportsViewsWithParameters() {
            return true;
          }
        };

    default boolean supportsInsert() {
      return true;
    }

    default boolean supportsUpsert() {
      return true;
    }

    default boolean supportsUpdate() {
      return true;
    }

    default boolean supportsDelete() {
      return true;
    }

    default boolean supportsParameters() {
      return true;
    }

    default boolean supportsViewsWithParameters() {
      return false;
    }

    default boolean supportsLock() {
      return true;
    }

    /**
     * The timestamp precision supported by this data store as number of fractional seconds
     *
     * @return the timestamp precision
     */
    default int timestampPrecision() {
      return 7;
    }
  }
}
