/*
 * © 2023-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds;

import com.sap.cds.CdsDataProcessor.Filter;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;
import java.util.Map;

/**
 * The CdsDiffProcessor allows to process two sets of nested maps of a CDS data that represent the
 * old and the new image of the data of the same type and notifies a {@link DiffVisitor} about
 * differences.
 */
public interface CdsDiffProcessor {

  Factory factory = Cds4jServiceLoader.load(Factory.class);

  /**
   * Creates a new CdsDiffProcessor instance.
   *
   * @return a DiffProcessor instance
   */
  static CdsDiffProcessor create() {
    return factory.create();
  }

  /**
   * Adds an object implementing interface {@link DiffVisitor}, which is notified about the
   * differences in all elements.
   *
   * @param diffVisitor the instance of the object that will observe found changes
   * @return this instance of the {@link CdsDiffProcessor}
   */
  default CdsDiffProcessor add(DiffVisitor diffVisitor) {
    return add((p, e, t) -> true, diffVisitor);
  }

  /**
   * Adds a {@link DiffVisitor}, which is notified about the differences in elements that match a
   * given filter.
   *
   * @param filter the filter predicate (see {@link Filter})
   * @param diffVisitor the visitor that will be notified about changes
   * @return this instance of the {@link CdsDiffProcessor}
   */
  CdsDiffProcessor add(CdsDataProcessor.Filter filter, DiffVisitor diffVisitor);

  /**
   * Runs the CdsDiffProcessor on the new and old image of the data.
   *
   * @param newImage new image of the data
   * @param oldImage old image of the data
   * @param type the type of the data
   */
  void process(Map<String, Object> newImage, Map<String, Object> oldImage, CdsStructuredType type);

  /**
   * Runs the CdsDiffProcessor on the new and old images of the data represented as a collection of
   * entries or an instances of the {@link com.sap.cds.Result}
   *
   * @param newImages sequence of the new images of the data
   * @param oldImages sequence of the old images of the data
   * @param type the type of the data
   */
  void process(
      Iterable<? extends Map<String, Object>> newImages,
      Iterable<? extends Map<String, Object>> oldImages,
      CdsStructuredType type);

  /**
   * A callback, which is called by the {@link CdsDiffProcessor} on differences between two images
   * of data.
   */
  interface DiffVisitor {

    /**
     * This method is called when the {@link CdsDiffProcessor} detects a change between values of an
     * element of an entity
     *
     * @param newPath the newPath contains the path to the element in the new image of the entity
     * @param oldPath the oldPath contains the path to the element in the old image state of the
     *     entity
     * @param element the {@link CdsElement} that represents the changed element
     * @param newValue the new value of the element or {@code null} if the element is removed or
     *     explicitly set to {@code null}
     * @param oldValue the old value of the element or {@code null} if the element is added
     */
    default void changed(
        Path newPath, Path oldPath, CdsElement element, Object newValue, Object oldValue) {}

    /**
     * This method is called when the {@link CdsDiffProcessor} detects that a new entry is added to
     * a collection of entities:
     *
     * <ul>
     *   <li>a composition of many entities
     *   <li>a cascading association to many entities
     *   <li>the images, if the processor processes a collection of images
     * </ul>
     *
     * <p>The method is called for every added entry.
     *
     * @param newPath path to the association where the change occurs in the new image of the
     *     entity.
     * @param oldPath path to the association where the change occurs in the old image of the
     *     entity.
     * @param association the instance of the {@link CdsElement} that represents element that
     *     defines the association where change occurs.
     * @param newValue the content of the added entity
     */
    default void added(
        Path newPath, Path oldPath, CdsElement association, Map<String, Object> newValue) {}

    /**
     * This method is called when the {@link CdsDiffProcessor} detects that an entry was removed
     * from the collection of entities:
     *
     * <ul>
     *   <li>a composition of many entities
     *   <li>a cascading association to many entities
     *   <li>the images, if the processor processes a collection of images
     * </ul>
     *
     * <p>The method is called for every removed entry.
     *
     * @param newPath path to the association where the change occurs in the new image of the
     *     entity.
     * @param oldPath path to the association where the change occurs in the old image of the
     *     entity.
     * @param association the instance of the {@link CdsElement} that represents element that
     *     defines the association where change occurs.
     * @param oldValue the content of the removed entity
     */
    default void removed(
        Path newPath, Path oldPath, CdsElement association, Map<String, Object> oldValue) {}
  }

  CdsDiffProcessor forDeepTraversal();

  interface Factory {

    CdsDiffProcessor create();
  }
}
