/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds;

import com.sap.cds.ql.Select;
import com.sap.cds.reflect.CdsStructuredType;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * Represents a result data thats typed with a {@link CdsData} class. {@link CdsResult} can be
 * serialized to JSON and accessed as an {@link Iterable} of type {@link D}.
 */
public interface CdsResult<D extends CdsData> extends Iterable<D>, JSONizable {

  /**
   * Creates a new typed {@link CdsResult} wrapper around the given {@link Result}.
   *
   * @param result the untyped {@link Result}
   * @param clazz the {@link CdsData} class to type the {@link Result} with
   */
  @SuppressWarnings("unchecked")
  public static <R extends CdsData> CdsResult<R> of(Result result, Class<R> clazz) {
    if (Row.class.equals(clazz)) {
      return (CdsResult<R>) result;
    }
    return new TypedResult<>(result, clazz);
  }

  /**
   * Returns the row of type {@link D}.
   *
   * @return an instance of type {@link D}
   * @throws EmptyResultException if the result is empty
   * @throws NonUniqueResultException if the result contains more than one row
   */
  D single();

  /**
   * Returns the single row mapped to the given {@code type}.
   *
   * @param <T> the return type
   * @param type the type the row is mapped to
   * @return a row mapped to {@code T}
   * @throws EmptyResultException if the result is empty
   * @throws NonUniqueResultException if the result contains more than one row
   */
  <T> T single(Class<T> type);

  /**
   * Returns the first row of type {@link D}.
   *
   * @return an optional {@link D}
   */
  Optional<D> first();

  /**
   * Returns the first row mapped to the given {@code type}.
   *
   * @param <T> the type of the {@code Optional}
   * @param type the type the row is mapped to
   * @return an {@code Optional} row mapped to {@code T}
   */
  <T> Optional<T> first(Class<T> type);

  /**
   * Returns a sequential {@code Stream} of the result rows with type {@link D}.
   *
   * @return a {@code Stream} of rows with type {@link D}.
   */
  Stream<D> stream();

  /**
   * Returns a sequential {@code Stream} of the result rows mapped to the given {@code type}.
   *
   * @param <T> the type of the {@code Stream}
   * @param type the type the result rows are mapped to
   * @return a {@code Stream} of the given {@code type}
   */
  <T> Stream<T> streamOf(Class<T> type);

  /**
   * Returns a list of the result rows with type {@link D}.
   *
   * @return a list of rows with type {@link D}.
   */
  List<D> list();

  /**
   * Returns a list of the result rows mapped to the given {@code type}.
   *
   * @param <T> the type of the {@code List}
   * @param type the type the result rows are mapped to
   * @return a list of the given {@code type}
   */
  <T> List<T> listOf(Class<T> type);

  /**
   * Returns the row type of this result.
   *
   * @return the row type
   */
  CdsStructuredType rowType();

  /**
   * Returns the number of rows affected or returned by the query. In case of a batch query the sum
   * over all batch row counts.
   *
   * @return the row count, -1 if undefined
   */
  long rowCount();

  /**
   * Returns the number of rows affected or returned by the query with the given batch index.
   *
   * @param batch the batch index
   * @return the row count of the batch, -1 if undefined
   * @throws IndexOutOfBoundsException if the batch index is out of range
   */
  long rowCount(int batch);

  /**
   * Returns the number of batches.
   *
   * @return the number of batches, -1 if undefined
   */
  int batchCount();

  /**
   * Returns the total number of root entities matching the query as requested by {@link
   * Select#inlineCount}.
   *
   * @return the row count, -1 if undefined
   */
  long inlineCount();
}
