/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds;

/**
 * Provides runtime configuration parameters for {@link CdsDataStore} implementations. The string
 * constants in this class are configuration keys consumed by implementations of this interface.
 */
public interface DataStoreConfiguration {

  // HANA
  String DOCSTORE_INTEGRATION_ENABLED = "cds.sql.hana.docstore";
  String IGNORE_LOCALE_ON_HANA = "cds.sql.hana.ignore-locale";
  String HANA_COMPATIBILITY_MODE = "cds.sql.hana.compatibilityMode";
  String HANA_COMPATIBILITY_MODE_LEGACY = "legacy";
  String HANA_COMPATIBILITY_MODE_HEX = "hex"; // default

  String COLLATE = "cds.sql.collate";
  String COLLATE_LOCALIZED = "localized-strings";

  String SEARCH_MODE = "cds.sql.search.mode";
  String SEARCH_MODE_GENERIC = "generic";
  String SEARCH_MODE_LOCALIZED_VIEW = "localized-view";
  String SEARCH_MODE_LOCALIZED_ASSOC = "localized-association";

  String SEARCH_HANA_FUZZY = "cds.sql.hana.search.fuzzy";
  String SEARCH_HANA_FUZZINESS = "cds.sql.hana.search.fuzzinessThreshold";
  String SEARCH_HANA_FUZZINESS_DEFAULT = "0.8";

  String INLINE_COUNT = "cds.sql.inlineCount";
  String INLINE_COUNT_AUTO = "auto";
  String INLINE_COUNT_QUERY = "query";
  String INLINE_COUNT_WINDOW_FUNCTION = "window-function";

  String RT_VIEW_MODE = "cds.sql.runtimeView.mode";
  String RT_VIEW_MODE_RESOLVE = "resolve";
  String RT_VIEW_MODE_CTE = "cte";

  String SELECT_STAR = "cds.sql.select.star";
  String SELECT_STAR_COLUMNS = "columns";

  String UNIVERSAL_CSN = "cds.model.universalCsn";

  // Don't log raw values of CQN queries by default
  String LOG_CQN_VALUES = "cds.ql.logging.log-values";

  String MAX_BATCH_SIZE = "cds.sql.max-batch-size";
  int MAX_BATCH_SIZE_DEFAULT = 1000;

  /**
   * Get the property value for a given key.
   *
   * @param key the key of the configuration property
   * @return the value for the provided key or null
   */
  String getProperty(String key);

  /**
   * Get the property value for a given key. If no value is present for the key it returns the
   * provided default value.
   *
   * @param key the key of the configuration property
   * @param defaultValue the default value if no value is present for the provided key
   * @return either the value for the provided key or the default value
   */
  default String getProperty(String key, String defaultValue) {
    String strVal = getProperty(key);

    return strVal != null ? strVal : defaultValue;
  }

  /**
   * Get the property value for a given key. If no value is present for the key it returns the
   * provided default value.
   *
   * @param key the key of the configuration property
   * @param defaultValue the default value if no value is present for the provided key
   * @return either the value for the provided key or the default value
   */
  default boolean getProperty(String key, boolean defaultValue) {
    String strVal = getProperty(key);

    return strVal != null ? Boolean.parseBoolean(strVal) : defaultValue;
  }

  /**
   * Get the property value for a given key. If no value is present for the key it returns the
   * provided default value.
   *
   * @param key the key of the configuration property
   * @param defaultValue the default value if no value is present for the provided key
   * @return either the value for the provided key or the default value
   */
  default int getProperty(String key, int defaultValue) {
    String strVal = getProperty(key);

    return strVal != null ? Integer.valueOf(strVal) : defaultValue;
  }
}
