/*
 * © 2018-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql;

import static java.util.Arrays.asList;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import com.google.common.annotations.Beta;
import com.sap.cds.CdsDataStore;
import com.sap.cds.CdsLockTimeoutException;
import com.sap.cds.CqnTableFunction;
import com.sap.cds.ql.cqn.CqnLock;
import com.sap.cds.ql.cqn.CqnParameter;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsEntity;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

/**
 * The Select builder allows to construct CDS QL select statements, which can be executed via the
 * {@link CdsDataStore}.
 *
 * <p>Note that some methods in this class mutually exclude each other and cannot be combined. For
 * example, the methods {@link #byId(Object) byId}, {@link #byParams(String...) byParams}, {@link
 * #matching(Map) matching}, and {@link #where(CqnPredicate) where} all set the where condition of
 * the enclosing statement, overwriting any previously stored conditions.
 *
 * @param <T> the type of the entity set that is the source of this select statement
 */
public abstract class Select<T extends StructuredType<?>> extends StatementBuilder<Select<T>>
    implements Source<T>, CqnSelect, FilterableStatement<T, Select<T>> {

  /**
   * Creates a select statement to select entries from a specified entity set.
   *
   * @param qualifiedName the fully qualified name of the CDS entity set
   * @return the select statement
   */
  public static Select<StructuredType<?>> from(String qualifiedName) {
    return CDS.QL.builder.select(qualifiedName);
  }

  /**
   * Creates a select statement to select entries from a specified entity set.
   *
   * @param ref the ref to the entity to select entries from
   * @return the select statement
   */
  public static Select<StructuredType<?>> from(CqnStructuredTypeRef ref) {
    return CDS.QL.builder.select(ref);
  }

  /**
   * Creates a select statement to select entries from a specified entity set.
   *
   * @param <T> the type of the entity set
   * @param entity the structured type representing the entity set
   * @return the select statement
   */
  public static <T extends StructuredType<T>> Select<T> from(StructuredType<T> entity) {
    return CDS.QL.builder.select(entity);
  }

  /**
   * Creates a select statement to select entries from a specified entity set.
   *
   * @param entityName the root entity name of the path expression
   * @param path a path expression navigating from the root entity to the target entity of the
   *     select statement
   * @return the select statement
   */
  public static Select<StructuredType<?>> from(
      String entityName, UnaryOperator<StructuredType<?>> path) {
    return CDS.QL.builder.select(entityName, path);
  }

  /**
   * Creates a select statement to select entries from a specified entity set.
   *
   * @param entity the model representation of the entity set obtained by reflection
   * @return the select statement
   * @see com.sap.cds.reflect.CdsModel#findEntity(String)
   * @see com.sap.cds.reflect.CdsModel#entities()
   */
  public static Select<StructuredType<?>> from(CdsEntity entity) {
    return CDS.QL.builder.select(entity);
  }

  /**
   * Creates a select statement to select entries from the result of a given subquery.
   *
   * @param subquery the subquery to select from
   * @return the select statement
   */
  public static Select<StructuredType<?>> from(CqnSelect subquery) {
    return CDS.QL.builder.select(subquery);
  }

  /**
   * Creates a select statement to select entries from a table function.
   *
   * @param tableFunction the table function
   * @return the select statement
   */
  @Beta
  public static Select<StructuredType<?>> from(CqnTableFunction tableFunction) {
    return CDS.QL.builder.select(tableFunction);
  }

  /**
   * Creates a select statement to select entries from a specified entity set.
   *
   * @param entity the model representation of the entity set obtained by reflection
   * @param path a path expression navigating from the root entity to the target entity of the
   *     select statement
   * @return the select statement
   * @see com.sap.cds.reflect.CdsModel#findEntity(String)
   * @see com.sap.cds.reflect.CdsModel#entities()
   */
  public static Select<StructuredType<?>> from(
      CdsEntity entity, UnaryOperator<StructuredType<?>> path) {
    return CDS.QL.builder.select(entity, path);
  }

  /**
   * Creates a select statement to select entries from a specified entity set.
   *
   * @param <T> the type of the entity set
   * @param entity the static model representation of the entity set
   * @return the select statement
   */
  public static <T extends StructuredType<T>> Select<T> from(Class<T> entity) {
    return CDS.QL.builder.select(entity);
  }

  /**
   * Creates a select statement to select entries from a specified entity set.
   *
   * @param <R> the type of the root entity
   * @param <T> the type of the target entity
   * @param entity the static model representation of the entity set
   * @param path a path expression navigating from the root entity to the target entity of the
   *     select statement
   * @return the select statement
   */
  public static <R extends StructuredType<R>, T extends StructuredType<T>> Select<T> from(
      Class<R> entity, Function<R, T> path) {
    return CDS.QL.builder.select(entity, path);
  }

  /**
   * Copies the given {@link CqnSelect} into a {@link Select} builder.
   *
   * @param select the {@code CqnSelect} to be copied
   * @return the modifiable select statement copy
   */
  public static Select<StructuredType<?>> copy(CqnSelect select) {
    return CDS.QL.builder.copy(select);
  }

  /**
   * Creates a select statement from a given CQN String.
   *
   * @param cqnSelect the CQN representation of the select statement
   * @return the select statement
   */
  public static Select<StructuredType<?>> cqn(String cqnSelect) {
    return CDS.QL.parser.select(cqnSelect);
  }

  /**
   * Specifies that duplicate query results should be eliminated.
   *
   * @return this select statement
   */
  public abstract Select<T> distinct();

  /**
   * Specifies that the total number of root entities matching this query shall be returned along
   * with the result.
   *
   * <p>Upon the calculation of the inline count the limit clause will be ignored.
   *
   * <p>Inline count must not be used in combination with distinct, groupy by or having.
   *
   * <p>The count may not exactly equal the actual number of root entities returned by this query.
   *
   * @return this select statement
   */
  public abstract Select<T> inlineCount();

  /**
   * Sets the projection of the select statement.
   *
   * @param items the {@link CqnSelectListItem CqnSelectListItem(s)} that shall be selected
   * @return this select statement
   */
  @SafeVarargs
  public final Select<T> columns(Selectable... items) {
    return columns(asList(items));
  }

  /**
   * Sets the projection of the select statement.
   *
   * @param items {@link Function Function(s)} providing the {@link CqnSelectListItem
   *     CqnSelectListItem(s)} that shall be selected
   * @return this select statement
   */
  @SafeVarargs
  public final Select<T> columns(Function<T, ? extends Selectable>... items) {
    return columns(applyFunctions(items));
  }

  /**
   * Sets the projection of the select statement.
   *
   * @param items the {@link CqnSelectListItem CqnSelectListItem(s)} that shall be selected
   * @return this select statement
   */
  public abstract Select<T> columns(List<? extends Selectable> items);

  /**
   * Sets the projection of the select statement.
   *
   * @param items the {@link CqnSelectListItem CqnSelectListItem(s)} that shall be selected
   * @return this select statement
   */
  public abstract Select<T> columns(Stream<? extends Selectable> items);

  /**
   * Sets the projection of the select statement. Elements are provided as Strings and can be paths
   * that refer to an element of an entity that is reached via an association or composition, e.g.
   * "author.name".
   *
   * @param elements the element(s) that shall be selected
   * @return this select statement
   */
  public abstract Select<T> columns(String... elements);

  /**
   * Sets the exclude list of the select statement.
   *
   * @param qualifiedNames the items that shall be excluded from the projection
   * @return this select statement
   */
  public final Select<T> excluding(String... qualifiedNames) {
    return excluding(asList(qualifiedNames));
  }

  /**
   * Sets the exclude list of the select statement.
   *
   * @param items {@link Function Function(s)} providing the {@link CqnSelectListValue
   *     CqnSelectListValue(s)} that shall be excluded from the projection
   * @return this select statement
   */
  @SafeVarargs
  public final Select<T> excluding(Function<T, CqnSelectListValue>... items) {
    return excluding(stream(items).map(f -> f.apply(getType()).displayName()).collect(toList()));
  }

  /**
   * Sets the exclude list of the select statement.
   *
   * @param qualifiedNames the items that shall be excluded from the projection
   * @return this select statement
   */
  public abstract Select<T> excluding(Collection<String> qualifiedNames);

  /**
   * Sets the where condition of this select statement. The where condition is provided as a
   * function that accepts a model object representing the entity targeted by the select and
   * returning the where condition.
   *
   * @param pred a {@link Function} providing the where condition
   * @return this select statement
   */
  @Override
  public abstract Select<T> where(Function<T, CqnPredicate> pred);

  /**
   * Sets the where condition of this select statement to a given predicate. If this statement
   * already has a where condition it is replaced with the given one.
   *
   * @param pred the {@link CqnPredicate}
   * @return this select statement
   */
  @Override
  public abstract Select<T> where(CqnPredicate pred);

  /**
   * Sets the where condition of the select statement. The where condition is computed from a map of
   * element names of the target entity set to their values, or a {@link CqnParameter}. The map
   * entries are transformed into equality predicates and joined via <b>and</b>.
   *
   * @param values the element name to value map defining the where condition
   * @return this select statement
   */
  @Override
  public abstract Select<T> matching(Map<String, ?> values);

  /**
   * Sets the where condition of the select statement. The where condition is computed from the
   * given vararg of element references of the target entity. For each element reference a predicate
   * is created comparing the element ref with a {@link CqnParameter} that has the name of the ref.
   * The predicates are joined via <b>and</b>.
   *
   * @param elementRefs the element references defining the where condition
   * @return this select statement
   */
  @Override
  public Select<T> byParams(String... elementRefs) {
    return byParams(asList(elementRefs).stream().collect(toSet()));
  }

  /**
   * Sets the where condition of the select statement. The where condition is computed from the
   * given collection of element references of the target entity. For each element ref a predicate
   * is created comparing the ref with a {@link CqnParameter} that has the name of the ref. The
   * predicates are joined via <b>and</b>.
   *
   * @param elementRefs the element references defining the where condition
   * @return this select statement
   */
  @Override
  public abstract Select<T> byParams(Collection<String> elementRefs);

  /**
   * Sets the where condition of the select statement, requiring that the value of the single key
   * element of the target entity set is equal to the given idValue.
   *
   * @param idValue the value of the target entity's key element
   * @return this select statement
   */
  @Override
  public abstract Select<T> byId(Object idValue);

  /**
   * Adds a search filter to this select statement, requiring that any searchable element of the
   * target entity set matches the given search term
   *
   * @param searchTerm the value to be searched for
   * @return this select statement
   */
  public abstract Select<T> search(String searchTerm);

  /**
   * Adds a search filter to this select statement, requiring that any searchable element of the
   * target entity set matches the given search predicate.
   *
   * @param pred search predicate
   * @return this select statement
   */
  public abstract Select<T> search(Function<Searchable, Predicate> pred);

  /**
   * Adds a search filter to this select statement, requiring that any searchable element of the
   * target entity set matches the given search predicate.
   *
   * @param pred search predicate
   * @return this select statement
   */
  public abstract Select<T> search(CqnPredicate pred);

  /**
   * Adds a search filter to this select statement, requiring that any searchable element of the
   * target entity set matches the given search predicate.
   *
   * @param pred search predicate
   * @param searchableElements elements to be searched in. Overwrites default searchable columns.
   * @return this select statement
   */
  public abstract Select<T> search(
      Function<Searchable, Predicate> pred, Iterable<String> searchableElements);

  /**
   * Adds a search filter to this select statement, requiring that any given searchable element of
   * the target entity set matches the given search term.
   *
   * @param searchTerm the value to be searched for
   * @param searchableElements elements to be searched in. Overwrites default searchable columns.
   * @return this select statement
   */
  public abstract Select<T> search(String searchTerm, Iterable<String> searchableElements);

  /**
   * Creates a write lock on the selected rows so that other queries cannot lock or change the data
   * until current transaction is finished.
   *
   * @param mode the lock mode
   * @param timeout maximum number of seconds to wait for [timeout] seconds for the lock
   *     acquisition. Afterwards the query execution will fail with a {@link
   *     CdsLockTimeoutException}.
   * @return this select statement
   */
  public abstract Select<T> lock(CqnLock.Mode mode, int timeout);

  /**
   * Creates a write lock on the selected rows for a default number of seconds specified by DB so
   * that other queries cannot lock or change the data until current transaction is finished. If a
   * lock cannot be acquired, the {@link CdsLockTimeoutException} is thrown upon statement
   * execution.
   *
   * @param mode the lock mode
   * @return this select statement
   */
  public abstract Select<T> lock(CqnLock.Mode mode);

  /**
   * Creates a write lock on the selected rows so that other queries cannot lock or change the data
   * until current transaction is finished.
   *
   * @param timeout maximum number of seconds to wait for [timeout] seconds for the lock
   *     acquisition. Afterwards the query execution will fail with the {@link
   *     CdsLockTimeoutException}.
   * @return this select statement
   */
  public Select<T> lock(int timeout) {
    return lock(CqnLock.Mode.EXCLUSIVE, timeout);
  }

  /**
   * Creates a write lock on the selected rows for a default number of seconds specified by DB so
   * that other queries cannot lock or change the data until current transaction is finished. If a
   * lock cannot be acquired, a {@link CdsLockTimeoutException} is thrown upon statement execution.
   *
   * @return this select statement
   */
  public Select<T> lock() {
    return lock(CqnLock.Mode.EXCLUSIVE);
  }

  /**
   * Sets the group by clause of the select statement.
   *
   * @param dimensions the elements the select shall be grouped by
   * @return this select statement
   */
  public final Select<T> groupBy(CqnValue... dimensions) {
    return groupBy(asList(dimensions));
  }

  /**
   * Sets the group by clause of the select statement.
   *
   * @param dimensions {@link Function Function(s)} providing the {@link CqnValue CqnValue(s)} the
   *     select shall be grouped by
   * @return this select statement
   */
  @SafeVarargs
  public final Select<T> groupBy(Function<T, CqnValue>... dimensions) {
    return groupBy(applyFunctions(dimensions));
  }

  /**
   * Sets the group by clause of the select statement.
   *
   * @param dimensions the {@link CqnValue CqnValue(s)} the select shall be grouped by
   * @return this select statement
   */
  public abstract Select<T> groupBy(List<? extends CqnValue> dimensions);

  /**
   * Sets the group by clause of the select statement.
   *
   * @param dimensions the {@link CqnValue CqnValue(s)} the select shall be grouped by
   * @return this select statement
   */
  public abstract Select<T> groupBy(Stream<? extends CqnValue> dimensions);

  /**
   * Sets the groupBy clause of the select statement. Elements are provided as Strings and can be
   * paths that refer to an element of an entity that is reached via an association or composition,
   * e.g. "author.name".
   *
   * @param elements the element(s) the select shall be grouped by
   * @return this select statement
   */
  public abstract Select<T> groupBy(String... elements);

  /**
   * Sets the having clause of the select statement.
   *
   * @param pred the having predicate condition
   * @return this select statement
   */
  public abstract Select<T> having(CqnPredicate pred);

  /**
   * Sets the having clause of the select statement.
   *
   * @param pred {@link Function Function(s)} providing the having predicate condition
   * @return this select statement
   */
  public abstract Select<T> having(Function<T, CqnPredicate> pred);

  /**
   * Sets the orderBy clause of the select statement.
   *
   * @param sortSpecs {@link Function Function(s)} providing the sort specification
   * @return this select statement
   */
  @SafeVarargs
  public final Select<T> orderBy(Function<T, CqnSortSpecification>... sortSpecs) {
    return orderBy(applyFunctions(sortSpecs));
  }

  /**
   * Sets the orderBy clause of the select statement.
   *
   * @param sortSpecs the sort specifications
   * @return this select statement
   */
  public final Select<T> orderBy(CqnSortSpecification... sortSpecs) {
    return orderBy(asList(sortSpecs));
  }

  /**
   * Sets the orderBy clause of the select statement.
   *
   * @param sortSpecs the sort specifications
   * @return this select statement
   */
  public abstract Select<T> orderBy(List<CqnSortSpecification> sortSpecs);

  /**
   * Sets the orderBy clause of the select statement with {@link CqnSelectListValue#displayName
   * display names} of select list values. The display names are provided as the strings and can be:
   *
   * <ul>
   *   <li>names of an elements
   *   <li>aliases of the {@link CqnSelectListValue select list values}
   *   <li>paths of display names that follow to-one associations
   *   <li>paths of display names that sub-elements of the structured elements
   * </ul>
   *
   * @param displayNames the display names defining the sort order
   * @return this select statement
   */
  public abstract Select<T> orderBy(String... displayNames);

  /**
   * Sets the limit of the select statement.
   *
   * @param top the maximum number of rows returned by the query or -1 if unlimited
   * @return this select statement
   */
  public Select<T> limit(long top) {
    return limit(top, 0l);
  }

  /**
   * Sets the limit and offset of the select statement.
   *
   * @param top the maximum number of rows returned by the query or -1 if unlimited
   * @param skip the number of rows that shall be skipped
   * @return this select statement
   */
  public abstract Select<T> limit(long top, long skip);

  @Override
  public String toString() {
    return toJson();
  }

  @SafeVarargs
  private final <R> List<R> applyFunctions(Function<T, ? extends R>... functions) {
    return stream(functions).map(f -> f.apply(getType())).collect(toList());
  }

  @Override
  public void accept(CqnVisitor visitor) {
    CqnSelect.super.accept(visitor);
  }
}
