/*
 * © 2023-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.ql.cqn.transformation;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.cqn.CqnElementRef;
import java.util.List;

/**
 * Transformation that groups the result set according to specified dimensions and applies given
 * transformations to each group.
 *
 * <p>The result set is the grouped input set after applying the transformations.
 *
 * <p>Two scenarios are common:
 *
 * <p>1. the list of transformations is empty -> the output set contains the distinct values of the
 * dimensions
 *
 * <p>2. a single {@link CqnAggregateTransformation aggregate} transformation is specified -> the
 * output set contains the aggregated values per dimension
 */
@Beta
public interface CqnGroupByTransformation extends CqnTransformation {

  /**
   * Returns the dimensions by which the input set is grouped.
   *
   * @return the dimensions by which the input set is grouped.
   */
  List<CqnElementRef> dimensions();

  /**
   * Returns the pipeline of transformations, which is applied to the grouped input set.
   *
   * @return the pipeline of transformations
   */
  default List<CqnTransformation> transformations() {
    return List.of();
  }

  @Override
  default Kind kind() {
    return Kind.GROUPBY;
  }

  @Override
  default void accept(CqnTransformationVisitor visitor) {
    visitor.visit(this);
  }
}
