/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.getReturnType;
import static com.sap.cds.generator.util.NamesUtils.qualifiedJavaClass;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.PUBLIC;

import java.util.function.Function;

import javax.lang.model.element.Modifier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.PoetTypeName;
import com.sap.cds.ql.CdsName;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsVisitor;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class CreateBuilderInterfaceVisitor implements CdsVisitor {

	private static final Logger logger = LoggerFactory.getLogger(CreateBuilderInterfaceVisitor.class);
	private static final String FILTER = "filter";
	private final TypeSpec.Builder entityClass;
	private final Configuration cfg;
	private String entityName;

	private CreateBuilderInterfaceVisitor(TypeSpec.Builder builder, Configuration cfg) {
		this.entityClass = builder;
		this.cfg = cfg;
	}

	public static CdsVisitor create(TypeSpec.Builder builder, Configuration cfg) {
		return new CreateBuilderInterfaceVisitor(builder, cfg);
	}

	@Override
	public void visit(CdsEntity entity) {
		String interfaceName = qualifiedJavaClass(cfg.getBasePackage(), entity.getQualifiedName())
				+ cfg.getClassNameSuffix();
		TypeName type = PoetTypeName.getTypeName(unqualifiedName(interfaceName));
		entityClass.addSuperinterface(ParameterizedTypeName.get(Types.STRUCTUREDTYPE, type));

		entityClass.addAnnotation(
				AnnotationSpec.builder(CdsName.class).addMember("value", "$S", entity.getQualifiedName()).build());
		addStaticQualifiedAttribute(entity);
		entityName = entity.getName();
	}

	@Override
	public void visit(CdsElement attribute) {
		if (!attribute.getType().isAssociation()) {
			addAttribute(attribute);
		} else {
			addAssociationAttribute(attribute);
		}
	}

	private void addStaticQualifiedAttribute(CdsEntity entity) {
		String qualifiedName = entity.getQualifiedName();

		FieldSpec staticField = FieldSpec.builder(String.class, "CDS_NAME")
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", qualifiedName)
				.build();
		entityClass.addField(staticField);
	}

	private void addAssociationAttribute(CdsElement attribute) {
		addMainFunction(attribute);
		addFilterFunction(attribute);
	}

	private void addAttribute(CdsElement attribute) {
		String name = attribute.getName();
		TypeName type = getReturnType(attribute, cfg);
		ParameterizedTypeName returnType = ParameterizedTypeName.get(Types.REFERENCE, type);
		entityClass
				.addMethod(MethodSpec.methodBuilder(name).addModifiers(PUBLIC, ABSTRACT).returns(returnType).build());
	}

	private void addMainFunction(CdsElement attribute) {
		CdsAssociationType association = attribute.getType();
		String targetEntityClassName = interfaceName(association.getTarget());
		TypeName returnType = PoetTypeName.getTypeName(targetEntityClassName);
		entityClass.addMethod(MethodSpec.methodBuilder(attribute.getName()).addModifiers(PUBLIC, ABSTRACT)
				.returns(returnType).build());

	}

	private void addFilterFunction(CdsElement attribute) {
		if (!attribute.getName().equalsIgnoreCase(FILTER)) {
			CdsAssociationType association = attribute.getType();
			String targetEntityClassName = interfaceName(association.getTarget());
			TypeName returnType = PoetTypeName.getTypeName(targetEntityClassName);
			entityClass
					.addMethod(MethodSpec.methodBuilder(attribute.getName())
							.addParameter(ParameterizedTypeName.get(ClassName.get(Function.class), returnType,
									Types.PREDICATE), "filter")
							.addModifiers(PUBLIC, ABSTRACT).returns(returnType).build());
		} else {
			logger.warn("There is a name clash between the element 'filter' of entity '" + entityName
					+ "' and a method in the super interface StructuredType. "
					+ "A filter function accepting predicates will not be generated.");
		}
	}

	public String interfaceName(CdsEntity entity) {
		return qualifiedJavaClass(cfg.getBasePackage(), entity.getQualifiedName()) + cfg.getClassNameSuffix();
	}
}
