/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.writer;

import static com.google.common.collect.Streams.concat;
import static com.sap.cds.generator.util.NamesUtils.checkForJavaKeyword;
import static com.sap.cds.generator.util.NamesUtils.javaPackage;
import static com.sap.cds.generator.util.NamesUtils.namespace;
import static com.sap.cds.generator.util.NamesUtils.qualifiedJavaClass;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedContextName;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.lang.model.element.Modifier;
import javax.tools.JavaFileObject;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.util.EntityWriterException;
import com.sap.cds.generator.util.GeneratedFile;
import com.sap.cds.generator.util.GeneratedFile.Consumer;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsAnnotation;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.TypeSpec;

public class ModelWriter implements CdsVisitor {

	private static final String CONTEXT_SUFFIX = "Context";
	private final Consumer consumer;
	private final Configuration config;
	private final CdsModel model;
	private final Map<String, CdsDefinition> wrapperMap;
	private final boolean isEventContext;

	public ModelWriter(Consumer consumer, Configuration config, CdsModel model) {
		this.consumer = consumer;
		this.config = config;
		this.model = model;
		this.wrapperMap = new HashMap<>();
		this.isEventContext = Boolean.parseBoolean(config.getEventContext());
	}

	@Override
	public void visit(CdsModel model) {
		wrapperMap.values().forEach(this::generateWrapperInterface);
	}

	@Override
	public void visit(CdsEntity entity) {
		checkForJavaKeyword(entity.getQualifiedName());
		if (!isExcluded(entity.getQualifiedName())) {
			collectWrapperInterfaces(entity);
			generateBuilderInterface(entity);
			generateConsumptionInterface(entity, entity.getQualifiedName());
			if (isEventContext) {
				concat(entity.actions(), entity.functions())
						.forEach(a -> generateEventContextInterface(a, entity.getQualifiedName(), true));
			}
		}
	}

	@Override
	public void visit(CdsEvent event) {
		checkForJavaKeyword(event.getQualifiedName());
		if (!isExcluded(event.getQualifiedName())) {
			collectWrapperInterfaces(event);
			generateConsumptionInterface(event, event.getQualifiedName());
			if (isEventContext) {
				generateEventContextInterface(event, event.getQualifiedName(), false);
			}
		}
	}

	@Override
	public void visit(CdsAction action) {
		if (isEventContext && !isExcluded(action.getQualifiedName())) {
			collectWrapperInterfaces(action);
			generateEventContextInterface(action, action.getQualifiedName(), false);
		}
	}

	@Override
	public void visit(CdsFunction function) {
		if (isEventContext && !isExcluded(function.getQualifiedName())) {
			collectWrapperInterfaces(function);
			generateEventContextInterface(function, function.getQualifiedName(), false);
		}
	}

	private boolean isExcluded(String qualifiedName) {
		List<String> excludes = config.getExcludes();
		boolean excluded = false;
		int i = qualifiedName.lastIndexOf('.');
		if (i > 0) {
			String namespace = qualifiedName.substring(0, i);
			excluded = excludes.stream().filter(e -> !e.contains("*")).anyMatch(e -> e.equals(namespace));
		}
		if (!excluded) {
			excluded = excludes.stream().filter(e -> e.contains("*")).anyMatch(qualifiedName::matches);
		}
		return excluded;
	}

	@Override
	public void visit(CdsStructuredType struct) {
		checkForJavaKeyword(struct.getQualifiedName());
		if (!isExcluded(struct.getQualifiedName())) {
			generateConsumptionInterface(struct, struct.getQualifiedName());
		}
	}

	private void collectWrapperInterfaces(CdsDefinition entity) {
		String packageName = javaPackage(config.getBasePackage(), entity.getQualifiedName());
		wrapperMap.put(packageName, entity);
	}

	private void generateWrapperInterface(CdsDefinition entity) {
		String qualifiedName = entity.getQualifiedName();
		String packageName = javaPackage(config.getBasePackage(), qualifiedName);
		String contextClassName = toUpperCamel(unqualifiedContextName(qualifiedName)) + config.getClassNameSuffix();

		TypeSpec.Builder builder = createInterfaceSpecBuilder(contextClassName);
		if (WrapperInterfaceCreator.create(builder, model, config).generateInterface(namespace(qualifiedName))) {
			writeType(packageName, builder.build());
		}
	}

	private void generateConsumptionInterface(CdsDefinition def, String qualifiedName) {
		TypeSpec.Builder builder = createInterfaceSpecBuilder(toUpperCamel(def.getName()));

		def.accept(CreateConsumptionInterfaceVisitor.create(builder, config, qualifiedName));

		Optional<CdsAnnotation<Object>> parentAnnotation = def
				.findAnnotation(CdsConstants.ANNOTATION_JAVA_EXTENDS.substring(1));
		parentAnnotation.ifPresent(annotation -> {
			@SuppressWarnings("unchecked")
			ArrayList<String> baseEntities = (ArrayList<String>) annotation.getValue();
			for (String baseEntityName : baseEntities) {
				CdsEntity baseEntity = model.getEntity(baseEntityName);
				String packageNameForBase = javaPackage(config.getBasePackage(), baseEntity.getQualifiedName());
				builder.addSuperinterface(ClassName.get(packageNameForBase, baseEntity.getName()));
			}
		});

		TypeSpec typeSpec = builder.build();
		String packageName = javaPackage(config.getBasePackage(), qualifiedName);
		writeType(packageName, typeSpec);
	}

	private void generateEventContextInterface(CdsDefinition def, String qualifiedName, boolean bounded) {
		TypeSpec.Builder builder = createInterfaceSpecBuilder(toUpperCamel(def.getName() + CONTEXT_SUFFIX));

		def.accept(CreateEventContextInterfaceVisitor.create(builder, config, qualifiedName, bounded));

		TypeSpec typeSpec = builder.build();
		String packageName = javaPackage(config.getBasePackage(), qualifiedName);
		writeType(packageName, typeSpec);
	}

	private void generateBuilderInterface(CdsEntity entity) {
		String interfaceName = qualifiedJavaClass(config.getBasePackage(), entity.getName())
				+ config.getClassNameSuffix();
		TypeSpec.Builder builder = createInterfaceSpecBuilder(interfaceName);

		entity.accept(CreateBuilderInterfaceVisitor.create(builder, config));

		TypeSpec typeSpec = builder.build();
		String packageName = javaPackage(config.getBasePackage(), entity.getQualifiedName());
		writeType(packageName, typeSpec);
	}

	private static TypeSpec.Builder createInterfaceSpecBuilder(String clazz) {
		String unqualifiedClassName = unqualifiedName(clazz);
		return TypeSpec.interfaceBuilder(unqualifiedClassName).addModifiers(Modifier.PUBLIC);
	}

	private void writeType(String packageName, TypeSpec typeSpec) {
		final JavaFile javaFile = JavaFile.builder(packageName, typeSpec).build();
		final JavaFileObject fileObject = javaFile.toJavaFileObject();
		GeneratedFile jpaFile = new GeneratedFile() {
			@Override
			public URI getUri() {
				return fileObject.toUri();
			}

			@Override
			public InputStream getContent() throws IOException {
				return fileObject.openInputStream();
			}
		};
		try {
			consumer.accept(jpaFile);
		} catch (IOException e) {
			String message = String.format("Exception while writing to file %s.", jpaFile.getUri());
			throw new EntityWriterException(message, e);
		}
	}
}
