package com.sap.cds.generator.util;

import static com.sap.cds.generator.util.PoetTypeName.getArrayTypeName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeFromCdsName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;

import java.util.List;
import java.util.Locale;

import javax.lang.model.SourceVersion;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.sap.cds.generator.Configuration;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.util.CdsModelUtils;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;

public class NamesUtils {

	private static final Logger logger = LoggerFactory.getLogger(NamesUtils.class);

	private NamesUtils() {
	}

	/**
	 * <p>
	 * Return package based on basePackage and qualifiedName.<br>
	 * Example:<br>
	 * a) basepackage = EMPTY; OUT = model<br>
	 * b) basepackage = EMPTY; qualifiedName = com.Model; OUT = com<br>
	 * c) basepackage = com, qualifiedName = Model; OUT = com<br>
	 * d) basepackage = com, qualifiedName = sap.Model; OUT = com.sap
	 * </p>
	 * 
	 * @param basePackage   the default package name prefix
	 * @param qualifiedName the fully qualified entity name
	 * @return String computed lowercase package name
	 * 
	 */
	public static String javaPackage(String basePackage, String qualifiedName) {
		String packageName = namespace(qualifiedName);
		if (!Strings.isNullOrEmpty(basePackage)) {
			packageName = Joiner.on('.').skipNulls().join(basePackage, packageName);
		}
		if (Strings.isNullOrEmpty(packageName)) {
			packageName = "model";
		}
		return packageName == null ? "" : packageName.toLowerCase(Locale.US);
	}

	public static String namespace(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');
		if (lastDot != -1) {
			return qualifiedName.substring(0, lastDot);
		}
		return null;
	}

	public static String unqualifiedName(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');

		return qualifiedName.substring(lastDot + 1);
	}

	public static String unqualifiedContextName(String qualifiedName) {
		if (qualifiedName.contains(".")) {
			String[] bits = qualifiedName.split("\\.");
			return bits[bits.length - 2];
		}
		return "CdsModel";
	}

	public static String qualifiedJavaClass(String basePackage, String entityName) {
		String packageName = javaPackage(basePackage, entityName);
		String name = unqualifiedName(entityName);

		return packageName + "." + toUpperCamel(name);
	}

	public static void checkForJavaKeyword(String qualifiedName) {
		int lastDot = qualifiedName.lastIndexOf('.');
		if (lastDot != -1) {
			String[] split = qualifiedName.split("\\.");
			for (String word : split) {
				if (SourceVersion.isKeyword(word)) {
					logger.warn("The Entity " + qualifiedName
							+ " contains a reserved Java keyword in its fully qualified name.");
				}
			}
		}
	}

	public static TypeName getAttributeType(CdsType type, Configuration cfg) {
		TypeName attributeType;

		if (type.isAssociation()) {
			String targetName = type.as(CdsAssociationType.class).getTarget().getQualifiedName();
			attributeType = getTypeFromCdsName(cfg.getBasePackage(), targetName);
			if (!CdsModelUtils.isSingleValued(type)) {
				attributeType = listOf(attributeType);
			}
		} else if (type.isSimple()) {
			attributeType = getTypeName(type.as(CdsSimpleType.class).getJavaType().getTypeName());
		} else if (type.isStructured()) {
			String targetName = type.getQualifiedName();
			attributeType = getTypeFromCdsName(cfg.getBasePackage(), targetName);
		} else if (type.isArrayed()) {
			CdsType itemsType = type.as(CdsArrayedType.class).getItemsType();
			TypeName innerAttributeType = getAttributeType(itemsType, cfg);
			if (innerAttributeType.toString().endsWith(".")) {
				// contains anonymous structured type with empty qualified name
				String className = toUpperCamel(unqualifiedName(type.getQualifiedName()));
				TypeName outer = getTypeFromCdsName(cfg.getBasePackage(), namespace(type.getQualifiedName()));
				attributeType = getArrayTypeName(getTypeName(outer + "." + className));
			} else {
				attributeType = getArrayTypeName(innerAttributeType);
			}
		} else {
			logger.warn("Interface Generation: Unsupported CDS Element with attribute name '"
					+ type.getName() + "' and type '" + type + "'");
			return null;
		}
		return attributeType;
	}

	public static TypeName getReturnType(CdsElement attribute, Configuration cfg) {
		boolean isMedia = attribute.annotations().anyMatch(a -> "Core.MediaType".equals(a.getName()));
		if (isMedia && attribute.getType().isSimple()) {
			CdsSimpleType simpleType = attribute.getType().as(CdsSimpleType.class);
			return PoetTypeName.getTypeName(CdsBaseType.cdsJavaMediaType(simpleType.getType()).getName());
		}
		if (attribute.getType().isStructured() && attribute.getType().getQualifiedName().isEmpty()) {
			return getTypeName(toUpperCamel(attribute.getName()));
		}
		return getAttributeType(attribute.getType(), cfg);
	}

	public static ParameterizedTypeName listOf(TypeName type) {
		return ParameterizedTypeName.get(ClassName.get(List.class), type);
	}
}
