/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.getReturnType;
import static com.sap.cds.generator.util.NamesUtils.listOf;
import static com.sap.cds.generator.util.NamesUtils.qualifiedJavaClass;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.writer.CaseFormatHelper.toLowerCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.SpecWriterUtil.addCdsNameAnnotation;
import static com.sap.cds.generator.writer.SpecWriterUtil.cdsNameAnnotation;
import static com.sap.cds.reflect.impl.CdsAnnotatableImpl.removeAt;
import static com.sap.cds.reflect.impl.reader.model.CdsConstants.ANNOTATION_CDS_JAVA_NAME;

import java.util.Map;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import com.sap.cds.CdsData;
import com.sap.cds.Struct;
import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.MethodStyle;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.util.CdsModelUtils;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.WildcardTypeName;

public class CreateConsumptionInterfaceVisitor implements CdsVisitor {

	private static final ParameterizedTypeName MAP_STR2OBJ = ParameterizedTypeName.get(ClassName.get(Map.class),
			ClassName.get(String.class), WildcardTypeName.subtypeOf(Object.class));
	private final TypeSpec.Builder builder;
	private final Configuration cfg;
	private final String typeName;

	public static CdsVisitor create(TypeSpec.Builder builder, Configuration config, String typeName) {
		return new CreateConsumptionInterfaceVisitor(builder, config, typeName);
	}

	private CreateConsumptionInterfaceVisitor(TypeSpec.Builder builder, Configuration config, String typeName) {
		this.builder = builder;
		this.cfg = config;
		this.typeName = qualifiedJavaClass(cfg.getBasePackage(), typeName);
		if (!builder.superinterfaces.contains(ClassName.get(CdsData.class))) {
			this.builder.addSuperinterface(CdsData.class);
		}
	}

	@Override
	public void visit(CdsEntity entity) {
		addRefMethod(entity);
		addStaticProxyMethod(getTypeName(toUpperCamel(entity.getName())));

		builder.addAnnotation(cdsNameAnnotation(entity.getQualifiedName(), "$S"));
	}

	private void addRefMethod(CdsEntity entity) {
		if (!entity.isAbstract()) {
			TypeName returnType = ClassName.bestGuess(toUpperCamel(entity.getName()) + "_");
			MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("ref").returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);

			builder.addMethod(methodBuilder.build());
		}
	}

	@Override
	public void visit(CdsEvent event) {
		addStaticProxyMethod(getTypeName(toUpperCamel(event.getName())));

		builder.addAnnotation(cdsNameAnnotation(event.getQualifiedName(), "$S"));
	}

	@Override
	public void visit(CdsStructuredType struct) {
		addStaticProxyMethod(getTypeName(toUpperCamel(struct.getName())));

		builder.addAnnotation(cdsNameAnnotation(struct.getQualifiedName(), "$S"));
	}

	private void addStaticProxyMethod(TypeName returnType, TypeSpec.Builder... anonymousbuilder) {
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		CodeBlock.Builder codeBuilder = CodeBlock.builder();
		codeBuilder.addStatement("return $T.create($T.class)", ClassName.get(Struct.class), returnType);

		methodBuilder.addCode(codeBuilder.build());
		if (anonymousbuilder.length > 0) {
			anonymousbuilder[0].addMethod(methodBuilder.build());
		} else {
			builder.addMethod(methodBuilder.build());
		}
	}

	@Override
	public void visit(CdsElement element) {
		SpecWriterUtil.addStaticField(builder, element);

		if (isAnonymousType(element)) {
			String unqualifiedClassName = unqualifiedName(toUpperCamel(element.getName()));
			TypeSpec.Builder innerIntefaceBuilder = TypeSpec.interfaceBuilder(unqualifiedClassName)
					.addModifiers(Modifier.PUBLIC, Modifier.STATIC);

			CdsType type = element.getType();
			Stream<CdsElement> elements;
			if (type.isStructured() && type.getQualifiedName().isEmpty()) {
				elements = element.getType().as(CdsStructuredType.class).elements();
			} else {
				elements = element.getType().as(CdsArrayedType.class).getItemsType().as(CdsStructuredType.class)
						.elements();
			}
			elements.forEach(e -> e
					.accept(CreateConsumptionInterfaceVisitor.create(innerIntefaceBuilder, cfg, element.getName())));
			addStaticProxyMethod(getTypeName(toUpperCamel(unqualifiedClassName)), innerIntefaceBuilder);
			populateGetter(element, getReturnType(element, cfg));
			populateSetter(element, getReturnType(element, cfg));
			builder.addType(innerIntefaceBuilder.build());
		} else {
			getter(element);
			setter(element);
		}

	}

	private boolean isAnonymousType(CdsElement element) {
		CdsType type = element.getType();
		if (type.isArrayed()) {
			CdsType itemsType = type.as(CdsArrayedType.class).getItemsType();
			if (itemsType.isStructured() && itemsType.getQualifiedName().isEmpty()) {
				return true;
			}
		} else if (type.isStructured() && type.getQualifiedName().isEmpty()) {
			return true;
		}
		return false;
	}

	private void setter(CdsElement attribute) {
		TypeName paramType = getSetterParam(attribute);

		if (paramType == null) {
			return;
		}
		populateSetter(attribute, paramType);
	}

	private void populateSetter(CdsElement attribute, TypeName paramType) {
		String setter;
		TypeName returnType;
		String attributeName = attribute.getAnnotationValue(removeAt(ANNOTATION_CDS_JAVA_NAME), attribute.getName());
		if (cfg.getMethodStyle() == MethodStyle.BEAN) {
			returnType = TypeName.VOID;
			setter = "set" + toUpperCamel(attributeName);
		} else {
			returnType = getTypeName(typeName);
			setter = toLowerCamel(attributeName);
		}

		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(setter).returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
				.addParameter(paramType, toLowerCamel(attributeName));
		addCdsNameAnnotation(methodBuilder, attribute.getName(), setter);
		builder.addMethod(methodBuilder.build());
	}

	private TypeName getSetterParam(CdsElement element) {
		if (element.getType().isAssociation()) {
			if (CdsModelUtils.isSingleValued(element.getType())) {
				return MAP_STR2OBJ;
			}
			return listOf(WildcardTypeName.subtypeOf(MAP_STR2OBJ));
		}
		return getReturnType(element, cfg);
	}

	private void getter(CdsElement attribute) {
		TypeName returnType = getReturnType(attribute, cfg);

		if (returnType == null) {
			return;
		}
		populateGetter(attribute, returnType);
	}

	private void populateGetter(CdsElement attribute, TypeName returnType) {
		String getter;
		String attributeName = attribute.getAnnotationValue(removeAt(ANNOTATION_CDS_JAVA_NAME), attribute.getName());
		if (cfg.getMethodStyle().equals(MethodStyle.BEAN)) {
			getter = "get" + toUpperCamel(attributeName);
		} else {
			getter = toLowerCamel(attributeName);
		}
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(getter).returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
		addCdsNameAnnotation(methodBuilder, attribute.getName(), getter);
		builder.addMethod(methodBuilder.build());
	}

}
