package com.sap.cds.generator.writer;

import static com.sap.cds.generator.util.NamesUtils.qualifiedJavaClass;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.util.TypeUtils.addStaticMethod;
import static com.sap.cds.generator.util.TypeUtils.getAttributeType;
import static com.sap.cds.generator.writer.CaseFormatHelper.lowercaseFirst;
import static com.sap.cds.generator.writer.CaseFormatHelper.toLowerCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperUnderscore;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.MethodStyle;
import com.sap.cds.generator.util.NamesUtils;
import com.sap.cds.ql.CdsName;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsParameter;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.util.CdsModelUtils;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.WildcardTypeName;

public class CreateEventContextInterfaceVisitor implements CdsVisitor {

	private static final String RETURN_TYPE = "ReturnType";
	private static final Logger logger = LoggerFactory.getLogger(CreateEventContextInterfaceVisitor.class);
	private static final ParameterizedTypeName MAP_STR2OBJ = ParameterizedTypeName.get(ClassName.get(Map.class),
			ClassName.get(String.class), WildcardTypeName.subtypeOf(Object.class));
	private static final String CDS_SERVICES_PACKAGE = "com.sap.cds.services";
	private static final String CQN_PACKAGE = "com.sap.cds.ql.cqn";
	private static final String CLASS_EVENT_CONTEXT = "EventContext";
	private static final String CLASS_EVENT_NAME = "EventName";
	private static final String CLASS_CQN_SELECT = "CqnSelect";
	private static final String CONTEXT_SUFFIX = "Context";
	private static final String PROXYMETHOD_ARG_NAME = "entityName";
	private final TypeSpec.Builder builder;
	private final Configuration cfg;
	private final boolean bounded;
	private final NamesUtils namesUtils;
	private final String boundEntityName;
	private final String definitionName;

	public static CdsVisitor create(TypeSpec.Builder builder, Configuration config, String boundEntityName,
			String definitionName, NamesUtils namesUtils) {
		return new CreateEventContextInterfaceVisitor(builder, config, boundEntityName, definitionName, namesUtils);
	}

	private CreateEventContextInterfaceVisitor(TypeSpec.Builder builder, Configuration config, String boundEntityName,
			String definitionName, NamesUtils namesUtils) {
		this.builder = builder;
		this.cfg = config;
		this.boundEntityName = boundEntityName;
		this.definitionName = qualifiedJavaClass(config.getBasePackage(), definitionName);
		this.builder.addSuperinterface(ClassName.get(CDS_SERVICES_PACKAGE, CLASS_EVENT_CONTEXT));
		this.bounded = boundEntityName != null ? true : false;
		this.namesUtils = namesUtils;
	}

	@Override
	public void visit(CdsAction action) {
		addGetCqnMethod();
		addSetCqnMethod();
		addStaticQualifiedAttribute(action);
		prepareDefinition(action);
	}

	@Override
	public void visit(CdsFunction function) {
		addGetCqnMethod();
		addSetCqnMethod();
		addStaticQualifiedAttribute(function);
		prepareDefinition(function);
	}

	@Override
	public void visit(CdsEvent event) {
		addGetter(event);
		addSetter(event);
		addStaticQualifiedAttribute(event);
		addStaticProxyMethod(event);
		builder.addAnnotation(eventNameAnnotation(event.getName(), "$S"));
	}

	private void addGetter(CdsEvent event) {
		String methodName;

		if (cfg.getMethodStyle() == MethodStyle.BEAN) {
			methodName = "getData";
		} else {
			methodName = "data";
		}
		MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(methodName)
				.returns(getTypeName(toUpperCamel(event.getName()))).addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
		builder.addMethod(resultGetterBuilder.build());
	}

	private void addSetter(CdsEvent event) {
		String methodName;
		TypeName returnType = TypeName.VOID;

		if (cfg.getMethodStyle() == MethodStyle.BEAN) {
			methodName = "setData";
		} else {
			methodName = "data";
			returnType = getTypeName(toUpperCamel(event.getName() + CONTEXT_SUFFIX));
		}
		MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(methodName).returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
				.addParameter(getTypeName(toUpperCamel(event.getName())), "event");
		builder.addMethod(resultGetterBuilder.build());
	}

	private void addGetCqnMethod() {
		if (bounded) {
			String methodName;

			if (cfg.getMethodStyle() == MethodStyle.BEAN) {
				methodName = "getCqn";
			} else {
				methodName = "cqn";
			}
			MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(methodName)
					.returns(ClassName.get(CQN_PACKAGE, CLASS_CQN_SELECT))
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
			builder.addMethod(resultGetterBuilder.build());
		}
	}

	private void addSetCqnMethod() {
		if (bounded) {
			String methodName = "cqn";
			TypeName returnType = ClassName.get(CQN_PACKAGE, CLASS_CQN_SELECT);

			if (cfg.getMethodStyle() == MethodStyle.BEAN) {
				returnType = TypeName.VOID;
				methodName = "setCqn";
			} else {
				returnType = getTypeName(toUpperCamel(unqualifiedName(definitionName) + CONTEXT_SUFFIX));
			}
			MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(methodName).returns(returnType)
					.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
					.addParameter(ClassName.get(CQN_PACKAGE, CLASS_CQN_SELECT), "select");
			builder.addMethod(resultGetterBuilder.build());
		}
	}

	private void addStaticQualifiedAttribute(CdsDefinition def) {
		String name = def.getName();

		FieldSpec staticField = FieldSpec.builder(String.class, "CDS_NAME")
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", name).build();
		builder.addField(staticField);
	}

	private void prepareDefinition(CdsDefinition def) {
		if (bounded) {
			addStaticProxyMethodBound(def);
		}
		addResultMethod(def);
		addStaticProxyMethod(def);
		builder.addAnnotation(eventNameAnnotation(def.getName(), "$S"));
	}

	private void addResultMethod(CdsDefinition def) {
		TypeName resultType = null, setReturnType;
		CdsType returnType;
		if (def instanceof CdsAction) {
			CdsAction action = (CdsAction) def;
			Optional<CdsType> actionType = action.returnType();
			if (actionType.isPresent()) {
				returnType = actionType.get();
			} else {
				return;
			}
		} else {
			CdsFunction function = (CdsFunction) def;
			returnType = function.getReturnType();
		}

		if (isAnonymousType(returnType)) {
			String unqualifiedClassName = RETURN_TYPE;
			TypeSpec.Builder innerIntefaceBuilder = TypeSpec.interfaceBuilder(unqualifiedClassName)
					.addModifiers(Modifier.PUBLIC, Modifier.STATIC);

			Stream<CdsElement> elements;
			if (returnType.isStructured() && returnType.getQualifiedName().isEmpty()) {
				elements = returnType.as(CdsStructuredType.class).elements();
			} else {
				elements = returnType.as(CdsArrayedType.class).getItemsType().as(CdsStructuredType.class).elements();
			}
			elements.forEach(e -> e
					.accept(CreateConsumptionInterfaceVisitor.create(innerIntefaceBuilder, cfg, unqualifiedClassName)));
			addStaticMethod(builder, getTypeName(toUpperCamel(unqualifiedClassName)), innerIntefaceBuilder);
			builder.addType(innerIntefaceBuilder.build());
			resultType = getTypeName(RETURN_TYPE);
		}
		if (resultType == null) {
			resultType = getResultType(def, returnType);
			if (resultType == null) {
				// Since actions might or might not return a value
				return;
			}
		}
		String setter, getter;
		if (cfg.getMethodStyle() == MethodStyle.BEAN) {
			setReturnType = TypeName.VOID;
			setter = "setResult";
			getter = "getResult";
		} else {
			setReturnType = getTypeName(toUpperCamel(def.getName() + CONTEXT_SUFFIX));
			setter = getter = "result";
		}

		MethodSpec.Builder resultSetterBuilder = MethodSpec.methodBuilder(setter).returns(setReturnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT).addParameter(resultType, toLowerCamel("result"));

		MethodSpec.Builder resultGetterBuilder = MethodSpec.methodBuilder(getter).returns(resultType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
		builder.addMethod(resultSetterBuilder.build());
		builder.addMethod(resultGetterBuilder.build());
	}

	private TypeName getResultType(CdsDefinition def, CdsType returnType) {
		if (returnType.isStructured()) {
			return getTypeName(namesUtils.qualifiedJavaClassName(returnType));
		} else if (returnType.isSimple()) {
			return getTypeName(returnType.as(CdsSimpleType.class).getJavaType().getTypeName());
		} else if (returnType.isArrayed()) {
			TypeName nestedType = getResultType(def, returnType.as(CdsArrayedType.class).getItemsType());
			return ParameterizedTypeName.get(ClassName.get(Collection.class), nestedType);
		} else {
			logger.warn("Consumption Interface Generation: Unsupported CDS Element with attribute name '"
					+ def.getName() + "' and type '" + returnType.getName() + "'");
			return null;
		}
	}

	private void addStaticProxyMethod(CdsDefinition entity) {
		TypeName returnType = getTypeName(toUpperCamel(entity.getName() + CONTEXT_SUFFIX));
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		if (bounded) {
			methodBuilder.addParameter(TypeName.get(String.class), PROXYMETHOD_ARG_NAME);
		}

		CodeBlock.Builder codeBuilder = CodeBlock.builder();
		codeBuilder.addStatement("return EventContext.create($T.class, $N)", returnType,
				bounded ? PROXYMETHOD_ARG_NAME : "null");

		methodBuilder.addCode(codeBuilder.build());
		builder.addMethod(methodBuilder.build());
	}

	private void addStaticProxyMethodBound(CdsDefinition entity) {
		TypeName returnType = getTypeName(toUpperCamel(entity.getName() + CONTEXT_SUFFIX));

		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		CodeBlock.Builder codeBuilder = CodeBlock.builder();
		codeBuilder.addStatement("return EventContext.create($T.class, $S)", returnType, this.boundEntityName);

		methodBuilder.addCode(codeBuilder.build());
		builder.addMethod(methodBuilder.build());
	}

	private static AnnotationSpec eventNameAnnotation(String cdsName, String format) {
		return AnnotationSpec.builder(ClassName.get(CDS_SERVICES_PACKAGE, CLASS_EVENT_NAME))
				.addMember("value", format, cdsName).build();
	}

	private boolean isAnonymousType(CdsType type) {
		if (type == null) {
			return false;
		} else if (type.isArrayed()) {
			return isAnonymousType(type.as(CdsArrayedType.class).getItemsType());
		} else if (type.isStructured() && type.getQualifiedName().isEmpty()) {
			return true;
		}
		return false;
	}

	@Override
	public void visit(CdsParameter parameter) {
		addStaticAttribute(parameter);
		addParamGetter(parameter);
		addParamSetter(parameter);
	}

	private void addParamGetter(CdsParameter parameter) {
		TypeName returnType = getAttributeType(parameter.getType(), cfg);
		String getter;

		if (returnType == null) {
			return;
		}
		if (cfg.getMethodStyle().equals(MethodStyle.BEAN)) {
			getter = "get" + toUpperCamel(parameter.getName());
		} else {
			getter = toLowerCamel(parameter.getName());
		}
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(getter).returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT);
		addCdsNameAnnotation(methodBuilder, parameter.getName(), getter);
		builder.addMethod(methodBuilder.build());

	}

	private void addParamSetter(CdsParameter parameter) {
		String setter;
		TypeName returnType;
		TypeName paramType = getSetterParam(parameter.getType());

		if (paramType == null) {
			return;
		}
		if (cfg.getMethodStyle() == MethodStyle.BEAN) {
			returnType = TypeName.VOID;
			setter = "set" + toUpperCamel(parameter.getName());
		} else {
			returnType = getTypeName(toUpperCamel(unqualifiedName(definitionName) + CONTEXT_SUFFIX));
			setter = toLowerCamel(parameter.getName());
		}

		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder(setter).returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.ABSTRACT)
				.addParameter(paramType, toLowerCamel(parameter.getName()));
		addCdsNameAnnotation(methodBuilder, parameter.getName(), setter);
		builder.addMethod(methodBuilder.build());

	}

	private TypeName getSetterParam(CdsType type) {
		if (type.isAssociation()) {
			if (CdsModelUtils.isSingleValued(type)) {
				return MAP_STR2OBJ;
			}
			return listOf(WildcardTypeName.subtypeOf(MAP_STR2OBJ));
		}
		return getAttributeType(type, cfg);
	}

	private static AnnotationSpec cdsNameAnnotation(String cdsName, String format) {
		return AnnotationSpec.builder(CdsName.class).addMember("value", format, cdsName).build();
	}

	private static MethodSpec.Builder addCdsNameAnnotation(MethodSpec.Builder builder, String cdsName,
			String javaName) {
		if (!cdsName.equals(propertyName(javaName))) {
			builder.addAnnotation(cdsNameAnnotation(toUpperUnderscore(cdsName), "$L"));
		}
		return builder;
	}

	private static String propertyName(String javaName) {
		if (javaName.startsWith("set") || javaName.startsWith("get")) {
			javaName = lowercaseFirst(javaName.substring(3));
		}
		return javaName;
	}

	private static ParameterizedTypeName listOf(TypeName type) {
		return ParameterizedTypeName.get(ClassName.get(List.class), type);
	}

	private void addStaticAttribute(CdsParameter parameter) {
		String name = toUpperUnderscore(parameter.getName());
		FieldSpec staticField = FieldSpec.builder(String.class, name)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL).initializer("$S", parameter.getName())
				.build();
		builder.addField(staticField);
	}
}
