/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.generator.util;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;

public class FileSystem implements GeneratedFile.Accessor {

	private final Path baseDirectory;
	private boolean mavenLayout;

	public FileSystem(Path baseDirectory, boolean mavenLayout) {
		this.mavenLayout = mavenLayout;
		File dir = baseDirectory.toFile();
		if (dir.exists() && !dir.isDirectory()) {
			String message = String.format("'%s' is not a directory.", baseDirectory.getFileName());
			throw new IllegalArgumentException(message);
		}
		this.baseDirectory = baseDirectory;
	}

	@Override
	public void accept(GeneratedFile file) throws IOException {
		try {
			Path outputPath = getPath(file);
			Files.createDirectories(outputPath.toFile().getParentFile().toPath());
			Files.copy(file.getContent(), outputPath, REPLACE_EXISTING);
		} catch (IOException e) {
			String message = String.format("Error writing file '%s' to file system.", file.getUri());
			throw new IOException(message, e);
		}
	}

	private Path getPath(FileLocation location) {
		Path base = getBase(location);
		Path outputPath = base.resolve(location.getUri().getPath());
		return outputPath;
	}

	private Path getBase(FileLocation location) {
		if (!mavenLayout) {
			return baseDirectory;
		}

		return baseDirectory.resolve(location.isResource() ? "resources" : "java");
	}

	@Override
	public GeneratedFile get(FileLocation location) throws IOException {
		Path path = getPath(location);

		return new GeneratedFile() {
			@Override
			public URI getUri() {
				return location.getUri();
			}

			@Override
			public InputStream getContent() throws IOException {
				if (!path.toFile().exists()) {
					return null;
				}
				return Files.newInputStream(path);
			}
		};
	}
}
