package com.sap.cds.generator.util;

import static com.sap.cds.generator.util.NamesUtils.namespace;
import static com.sap.cds.generator.util.NamesUtils.qualifiedJavaClass;
import static com.sap.cds.generator.util.NamesUtils.unqualifiedName;
import static com.sap.cds.generator.util.PoetTypeName.getArrayTypeName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeFromCdsName;
import static com.sap.cds.generator.util.PoetTypeName.getTypeName;
import static com.sap.cds.generator.writer.CaseFormatHelper.toUpperCamel;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.Struct;
import com.sap.cds.generator.Configuration;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.util.CdsModelUtils;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class TypeUtils {

	private static final Logger logger = LoggerFactory.getLogger(TypeUtils.class);

	private TypeUtils() {
	}

	public static TypeName getAttributeType(CdsType type, Configuration cfg) {
		TypeName attributeType;

		if (type.isAssociation()) {
			CdsAssociationType assocType = type.as(CdsAssociationType.class);
			CdsType target = assocType.getTargetAspect().orElseGet(assocType::getTarget);
			attributeType = getTypeFromCdsName(cfg.getBasePackage(), target.getQualifiedName(), target.getName());
			if (!CdsModelUtils.isSingleValued(type)) {
				attributeType = listOf(attributeType);
			}
		} else if (type.isSimple()) {
			attributeType = getTypeName(type.as(CdsSimpleType.class).getJavaType().getTypeName());
		} else if (type.isStructured()) {
			String targetName = type.getQualifiedName();
			attributeType = getTypeFromCdsName(cfg.getBasePackage(), targetName, type.getName());
		} else if (type.isArrayed()) {
			CdsType itemsType = type.as(CdsArrayedType.class).getItemsType();
			TypeName innerAttributeType = getAttributeType(itemsType, cfg);
			if (innerAttributeType.toString().endsWith(".")) {
				// contains anonymous structured type with empty qualified name
				String className = toUpperCamel(unqualifiedName(type.getQualifiedName()));
				TypeName outer = getTypeName(
						qualifiedJavaClass(cfg.getBasePackage(), namespace(type.getQualifiedName())));
				attributeType = getArrayTypeName(getTypeName(outer + "." + className));
			} else {
				attributeType = getArrayTypeName(innerAttributeType);
			}
		} else {
			logger.warn("Interface Generation: Unsupported CDS Element with attribute name '" + type.getName()
					+ "' and type '" + type + "'");
			return null;
		}
		return attributeType;
	}

	public static TypeName getReturnType(CdsElement attribute, Configuration cfg) {
		boolean isMedia = attribute.annotations().anyMatch(a -> "Core.MediaType".equals(a.getName()));
		if (isMedia && attribute.getType().isSimple()) {
			CdsSimpleType simpleType = attribute.getType().as(CdsSimpleType.class);
			return PoetTypeName.getTypeName(CdsBaseType.cdsJavaMediaType(simpleType.getType()).getName());
		}
		if (attribute.getType().isStructured() && attribute.getType().getQualifiedName().isEmpty()) {
			return getTypeName(toUpperCamel(attribute.getName()));
		} else if (isAnonymousAspect(attribute)) {
			TypeName attributeType;
			attributeType = getTypeName(toUpperCamel(attribute.getName()));
			if (!CdsModelUtils.isSingleValued(attribute.getType())) {
				attributeType = listOf(attributeType);
			}
			return attributeType;
		}
		return getAttributeType(attribute.getType(), cfg);
	}

	public static ParameterizedTypeName listOf(TypeName type) {
		return ParameterizedTypeName.get(ClassName.get(List.class), type);
	}

	public static ClassName className(String name) {
		return ClassName.bestGuess(toUpperCamel(name));
	}

	public static ClassName builderClassName(CdsDefinition def) {
		return ClassName.bestGuess(toUpperCamel(def.getName()) + "_");
	}

	public static boolean isAnonymousAspect(CdsElement element) {
		CdsType type = element.getType();
		if (type.isAssociation()) {
			Optional<CdsType> targetAspect = type.as(CdsAssociationType.class).getTargetAspect();
			if (targetAspect.isPresent() && targetAspect.get().getQualifiedName().isEmpty()) {
				return true;
			}
		}
		return false;
	}

	public static Stream<CdsElement> getAnonymousElements(CdsElement element) {
		CdsType type = element.getType();
		if (type.isAssociation()) {
			Optional<CdsType> targetAspect = type.as(CdsAssociationType.class).getTargetAspect();
			if (targetAspect.isPresent() && targetAspect.get().getQualifiedName().isEmpty()) {
				return targetAspect.get().as(CdsStructuredType.class).elements();
			}
		}
		return Stream.empty();
	}

	public static void addStaticMethod(TypeSpec.Builder builder, TypeName returnType,
			TypeSpec.Builder... anonymousbuilder) {
		MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("create").returns(returnType)
				.addModifiers(Modifier.PUBLIC, Modifier.STATIC);
		CodeBlock.Builder codeBuilder = CodeBlock.builder();
		codeBuilder.addStatement("return $T.create($T.class)", ClassName.get(Struct.class), returnType);

		methodBuilder.addCode(codeBuilder.build());
		if (anonymousbuilder.length > 0) {
			anonymousbuilder[0].addMethod(methodBuilder.build());
		} else {
			builder.addMethod(methodBuilder.build());
		}
	}
}
